use std::thread::sleep;
use std::time::Duration;
use std::cell::{Cell, RefCell};

use chrono::offset::Local;

#[cfg(any(
    target_os = "freebsd",
    target_os = "dragonfly",
    target_os = "openbsd",
    target_os = "netbsd",
    target_os = "macos"
))]
use chrono::SecondsFormat;

use super::portable;
use super::socket::*;
use super::common::*;
use super::error::{SyRes/* , SyslogError*/};

/// A common instance which describes the syslog state
pub struct SyncSyslogInternal
{   
    /// A identification i.e program name, thread name
    logtag: RefCell<Option<String>>, 

    /// Defines how syslog operates
    logstat: LogStat,

    /// Holds the facility 
    facility: LogFacility,

    /// A logmask
    logmask: Cell<i32>,

    /// A stream
    stream: RefCell<SyslogSocket>,
}

impl Drop for SyncSyslogInternal 
{
    fn drop(&mut self) 
    {
        self.disconnectlog();
    }
}

/// Private realization. It is assumed that functions which are called,
/// are called after 'lock' is locked.
impl SyncSyslogInternal
{
    /// Creates new instance of [SyslogOption].
    ///
    /// # Arguments
    ///
    /// * `ident` - An optional argument which takes ref to str. If none, the
    ///     ident will be set later. Yje ident will be trucated to 48 UTF8
    ///     chars.
    /// 
    /// * `logstat` - A [LogStat] flags separated by '|'
    /// 
    /// * `facility` - A [LogFacility] flag
    ///
    /// # Returns
    ///
    /// * The instance of the [SyslogOption]. Never fails.
    pub(crate) fn new(
        ident: Option<&str>, 
        logstat: LogStat, 
        facility: LogFacility
    ) -> Self
    {
        // check if log_facility is invalid
        let log_facility =
            if facility.is_empty() == false && 
                (facility & !LogMask::LOG_FACMASK).is_empty() == true
            {
                facility
            }
            else
            {
                // default facility code
                LogFacility::LOG_USER
            };

        let ident = match ident
            {
                Some(r) => Some(truncate_n(r, 48)),
                None => None,
            };
        
        return 
            Self
            {
                logtag: RefCell::new(ident),
                logstat: logstat, 
                facility: log_facility,
                logmask: Cell::new(0xff),
                stream: RefCell::new(SyslogSocket::none()),
            };
    }

    fn send_to_stderr(&self, msg: &mut [u8])
    {
        if self.logstat.intersects(LogStat::LOG_PERROR) == true
        {
            let mut newline = String::from("\n");
            send_to_stderr(libc::STDERR_FILENO, msg, &mut newline);
        }
    }

    fn is_logmasked(&self, pri: i32) -> bool
    {
        if ((1 << (pri & LogMask::LOG_PRIMASK)) & self.logmask.get()) == 0
        {
            return true;
        }

        return false;
    }

    pub(crate) 
    fn set_logmask(&self, logmask: i32) -> i32
    {
        let oldmask = self.logmask.get();

        if logmask != 0
        {
            self.logmask.set(logmask);
        }

        return oldmask;
    }

    pub(crate) fn set_logtag<L: AsRef<str>>(&self, logtag: L)
    {
        *self.logtag.borrow_mut() = 
            Some(truncate_n(logtag.as_ref(), 48));
    }

    /// Disconnects the unix stream from syslog.
    /// Should be called only when lock is acuired
    pub(crate) fn disconnectlog(&self)
    {
        if self.stream.borrow().is_none() == false
        {
            self.stream.borrow_mut().shutdown();

            *self.stream.borrow_mut() = SyslogSocket::none();
        }
    }

    /// Connects unix stream to the syslog and sets up the properties of
    /// the unix stream.
    /// Should be called only when lock is acuired
    pub(crate) fn connectlog(&self) -> SyRes<()>
    {
        let stream = SyslogSocket::connect()?;

        *self.stream.borrow_mut() = stream;

        return Ok(());
    }

    /// An internal function which is called by the syslog or vsyslog.
    /// A glibc implementation RFC3164
    #[cfg(target_os = "linux")]
    pub(crate) fn vsyslog1<S: AsRef<str>>(&self, mut pri: i32, fmt: S)
    {
        // check for invalid bits
        match check_invalid_bits(&mut pri)
        {
            Ok(_) => {},
            Err(_e) => self.vsyslog1(get_internal_log(), fmt.as_ref())
        }

        // check priority against setlogmask
        if self.is_logmasked(pri) == true
        {
            return;
        }

        // set default facility if not specified in pri
        if (pri & LOG_FACMASK) == 0
        {
            pri |= self.facility.bits();
        }

        // get timedate
        let timedate = Local::now().format("%h %e %T").to_string();

        // get appname
        if self.logtag.borrow().is_none() == true
        {
            match portable::p_getprogname()
            {
                Some(r) => 
                {
                    *self.logtag.borrow_mut() = 
                        Some(truncate_n(r.as_str(), 48));
                },
                None => 
                {
                    *self.logtag.borrow_mut() = 
                        Some(truncate_n("unknown", 48));
                }
            }
        }

        let b_progname = self.logtag.borrow();
        let progname = b_progname.as_ref().unwrap();
        

        let msg = fmt.as_ref();
        let msg_final = 
            if msg.ends_with("\n") == true
            {
                truncate(msg)
            }
            else
            {
                msg
            };

        // message based on RFC 3164
        let msg_pri = 
            [
                b"<", pri.to_string().as_bytes(), b">"
            ].concat();

        let msg_header = 
            [ 
                // timedate
                timedate.as_bytes(), 
                // hostname
               // " ".as_bytes(), hostname.as_bytes(), 
            ].concat();

        let mut msg = 
            [
                // appname
                b" ", progname.as_bytes(), 
                // PID
                b"[", portable::get_pid().to_string().as_str().as_bytes(), b"]:",
                // msg
                b" ", /*b"\xEF\xBB\xBF",*/ msg_final.as_bytes()
            ].concat();
        
        drop(b_progname);

        // output to stderr if required
        self.send_to_stderr(&mut msg);
        
        
        let fullmsg = [msg_pri.as_slice(), msg_header.as_slice(), msg.as_slice()].concat();

        if self.stream.borrow().is_none() == true
        {
            // open connection
            match self.connectlog()
            {
                Ok(_) => {},
                Err(e) =>
                {
                    self.send_to_stderr(unsafe { e.eject_string().as_bytes_mut() } );
                    return;
                }
            }
        }

        // There are two possible scenarios when send may fail:
        // 1. syslog temporary unavailable
        // 2. syslog out of buffer space
        // If we are connected to priv socket then in case of 1 we reopen connection
        //      and retry once.
        // If we are connected to unpriv then in case of 2 repeatedly retrying to send
        //      until syslog socket buffer space will be cleared

        loop
        {
            let mut stream = self.stream.borrow_mut();

            match stream.send(&fullmsg)
            {
                Ok(_) => return,
                Err(err) =>
                {   
                    if let Some(libc::ENOBUFS) = err.raw_os_error()
                    {
                        // scenario 2
                        if stream.is_priv() == true
                        {
                            break;
                        }

                        sleep(Duration::from_micros(1));
                        drop(stream);
                    }
                    else
                    {
                        // scenario 1
                        drop(stream);

                        self.disconnectlog();
                        match self.connectlog()
                        {
                            Ok(_) => {},
                            Err(_e) => break,
                        }

                        // if resend will fail then probably the scn 2 will take place
                    }   
                }
            }
        } // loop


        // If program reached this point then transmission over socket failed.
        // Try to output message to console

        if self.logstat.intersects(LogStat::LOG_CONS)
        {
            let fd = unsafe {
                libc::open(
                    PATH_CONSOLE.as_ptr(), 
                    libc::O_WRONLY | libc::O_NONBLOCK | libc::O_CLOEXEC, 
                    0
                )
            };

            if fd >= 0
            {
                let mut without_pri = [msg_header.as_slice(), msg.as_slice()].concat();
                let mut newline = String::from("\r\n");
                send_to_stderr(fd, without_pri.as_mut_slice(),&mut newline);

                unsafe {libc::close(fd)};
            }
        }
    }

    /// An internal function which is called by the syslog or vsyslog.
    /// A glibc implementation RFC5424
    #[cfg(any(
        target_os = "freebsd",
        target_os = "dragonfly",
        target_os = "openbsd",
        target_os = "netbsd",
        target_os = "macos"
    ))]
    pub(crate) fn vsyslog1<S: AsRef<str>>(&self, mut pri: i32, fmt: S)
    {
        // check for invalid bits
        match check_invalid_bits(&mut pri)
        {
            Ok(_) => {},
            Err(_e) => self.vsyslog1(get_internal_log(), fmt.as_ref())
        }

        // check priority against setlogmask
        if self.is_logmasked(pri) == true
        {
            return;
        }

        // set default facility if not specified in pri
        if (pri & LOG_FACMASK) == 0
        {
            pri |= self.facility.bits();
        }

        let mut hostname_buf = [0u8; MAXHOSTNAMELEN];
        let hostname = 
            match nix::unistd::gethostname(&mut hostname_buf)
            {
                Ok(r) =>
                {
                    match r.to_str()
                    {
                        Ok(r) => r,
                        Err(_e) => NILVALUE
                    }
                },
                Err(_e) => NILVALUE,
            };

        // get timedate
        let timedate = 
            Local::now().to_rfc3339_opts(SecondsFormat::Secs, false);

        // get appname
        if self.logtag.borrow().is_none() == true
        {
            match portable::p_getprogname()
            {
                Some(r) => 
                {
                    *self.logtag.borrow_mut() = 
                        Some(truncate_n(r.as_str(), 48));
                },
                None => 
                {
                    *self.logtag.borrow_mut() = Some(NILVALUE.to_string());
                }
            }
        }

        let b_progname = self.logtag.borrow();
        let progname = b_progname.as_ref().unwrap();
        

        let msg = fmt.as_ref();
        let msg_final = 
            if msg.ends_with("\n") == true
            {
                truncate(msg)
            }
            else
            {
                msg
            };

        // message based on RFC 5424
        let msg_pri = 
            [
                b"<", pri.to_string().as_bytes(), b">1"
            ].concat();

        let msg_header = 
            [ 
                // timedate
                b" ", timedate.as_bytes(), 
                // hostname
                b" ", hostname.as_bytes(), 
            ].concat();
        
        let mut msg = 
            [
                // appname
                b" ", progname.as_bytes(), 
                // PID
                b" ", portable::get_pid().to_string().as_str().as_bytes(),
                // message ID
                b" ", NILVALUE.as_bytes(), 
                // structured data
                b" ", NILVALUE.as_bytes(), 
                // msg
                b" ", /*b"\xEF\xBB\xBF",*/ msg_final.as_bytes()
            ].concat();
        
        drop(progname);

        // output to stderr if required
        self.send_to_stderr(&mut msg);
        
        
        let fullmsg = [msg_pri.as_slice(), msg_header.as_slice(), msg.as_slice()].concat();

        if self.stream.borrow().is_none() == true
        {
            // open connection
            match self.connectlog()
            {
                Ok(_) => {},
                Err(e) =>
                {
                    self.send_to_stderr(unsafe { e.eject_string().as_bytes_mut() } );
                    return;
                }
            }
        }

        // There are two possible scenarios when send may fail:
        // 1. syslog temporary unavailable
        // 2. syslog out of buffer space
        // If we are connected to priv socket then in case of 1 we reopen connection
        //      and retry once.
        // If we are connected to unpriv then in case of 2 repeatedly retrying to send
        //      until syslog socket buffer space will be cleared

        loop
        {
            let mut stream = self.stream.borrow_mut();

            match stream.send(&fullmsg)
            {
                Ok(_) => return,
                Err(err) =>
                {   
                    if let Some(libc::ENOBUFS) = err.raw_os_error()
                    {
                        // scenario 2
                        if stream.is_priv() == true
                        {
                            break;
                        }

                        sleep(Duration::from_micros(1));
                        drop(stream);
                    }
                    else
                    {
                        // scenario 1
                        drop(stream);

                        self.disconnectlog();
                        match self.connectlog()
                        {
                            Ok(_) => {},
                            Err(_e) => break,
                        }

                        // if resend will fail then probably the scn 2 will take place
                    }   
                }
            }
        } // loop


        // If program reached this point then transmission over socket failed.
        // Try to output message to console

        if self.logstat.intersects(LogStat::LOG_CONS)
        {
            let fd = unsafe {
                libc::open(
                    PATH_CONSOLE.as_ptr(), 
                    libc::O_WRONLY | libc::O_NONBLOCK | libc::O_CLOEXEC, 
                    0
                )
            };

            if fd >= 0
            {
                let mut without_pri = [msg_header.as_slice(), msg.as_slice()].concat();
                let mut newline = String::from("\r\n");
                send_to_stderr(fd, without_pri.as_mut_slice(),&mut newline);

                unsafe {libc::close(fd)};
            }
        }

    }
}

#[test]
fn test_bit_operations()
{
    let correct = 
        SyncSyslogInternal::new(Some("test1"), LogStat::LOG_PID, LogFacility::LOG_DAEMON);

    assert_eq!(correct.facility, LogFacility::LOG_DAEMON);
    assert_eq!((correct.facility & !LogFacility::LOG_DAEMON), LogFacility::empty());
}

#[test]
fn test_bit_operations2()
{
    let _correct = 
        SyncSyslogInternal::new(Some("test2"), LogStat::LOG_PID, LogFacility::LOG_DAEMON);

    let mut pri = Priority::LOG_ALERT.bits();

    let res = check_invalid_bits(&mut pri);

    assert_eq!(res.is_ok(), true);
    assert_eq!(pri, Priority::LOG_ALERT.bits());
}

#[test]
fn test_set_priority()
{
    use super::LOG_MASK;

    let correct = 
        SyncSyslogInternal::new(Some("test1"), LogStat::LOG_PID, LogFacility::LOG_DAEMON);

    let ret = correct.set_logmask(LOG_MASK!(Priority::LOG_ERR));

    assert_eq!(ret, 0xff);

    let ret = correct.set_logmask(LOG_MASK!(Priority::LOG_ERR));

    assert_eq!(ret, LOG_MASK!(Priority::LOG_ERR));

    let ret = correct.is_logmasked(Priority::LOG_WARNING.bits());
    assert_eq!(ret, true);

    let ret = correct.is_logmasked(Priority::LOG_ERR.bits());
    assert_eq!(ret, false);

    let ret = correct.is_logmasked(Priority::LOG_CRIT.bits());
    assert_eq!(ret, true);
}

#[test]
fn test_set_priority2()
{
    use super::LOG_MASK;
    
    let correct = 
        SyncSyslogInternal::new(Some("test1"), LogStat::LOG_PID, LogFacility::LOG_DAEMON);

    let ret = correct.set_logmask(!LOG_MASK!(Priority::LOG_ERR));

    assert_eq!(ret, 0xff);

    let ret = correct.is_logmasked(Priority::LOG_WARNING.bits());
    assert_eq!(ret, false);

    let ret = correct.is_logmasked(Priority::LOG_ERR.bits());
    assert_eq!(ret, true);

    let ret = correct.is_logmasked(Priority::LOG_CRIT.bits());
    assert_eq!(ret, false);
}
