/*-
* syslog-rs - a syslog client translated from libc to rust
* Copyright (C) 2021  Aleksandr Morozov
* 
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU Lesser General Public
* License as published by the Free Software Foundation; either
* version 3 of the License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
* Lesser General Public License for more details.
* 
* You should have received a copy of the GNU Lesser General Public License
* along with this program; if not, write to the Free Software Foundation,
* Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*/

use std::{
    mem::MaybeUninit, 
};
use std::sync::atomic::{AtomicBool, Ordering};
use std::cell::RefCell;

use crossbeam::utils::{Backoff, CachePadded};

use chrono::offset::Local;

#[cfg(any(
    target_os = "freebsd",
    target_os = "dragonfly",
    target_os = "openbsd",
    target_os = "netbsd",
    target_os = "macos"
))]
use chrono::SecondsFormat;

//use crate::{map_error, throw_error};

use super::socket::*;
use super::portable;
use super::common::*;
use super::error::{SyRes/* , SyslogError*/};




/// A common instance which describes the syslog state
pub struct Syslog
{   
    /// A multithreading sync lock. A giant lock which used for access
    /// to both - stream and [SyslogOption]
    lock: AtomicBool,

    /// A [CachePadded] [SyslogOption] instance which is mostly read only.
    option: CachePadded<SyslogOption>,

    /// A stream
    stream: RefCell<SyslogSocket>,
}

unsafe impl Send for Syslog {}
unsafe impl Sync for Syslog {}

impl Drop for Syslog 
{
    fn drop(&mut self) 
    {
        self.disconnectlog();
    }
}

/// Private realization. It is assumed that functions which are called,
/// are called after 'lock' is locked.
impl Syslog
{
    /// Disconnects the unix stream from syslog.
    /// Should be called only when lock is acuired
    fn disconnectlog(&self)
    {
        if self.stream.borrow().is_none() == false
        {
            self.stream.borrow_mut().shutdown();

            *self.stream.borrow_mut() = SyslogSocket::none();
        }
    }

    /// Connects unix stream to the syslog and sets up the properties of
    /// the unix stream.
    /// Should be called only when lock is acuired
    fn connectlog(&self) -> SyRes<()>
    {
        // try priv socket
        let stream = SyslogSocket::connect()?;

        let mut len: MaybeUninit<libc::socklen_t> = std::mem::MaybeUninit::uninit();
        // set the sndbuf len
        let res = unsafe
            {
                libc::getsockopt(
                    stream.get_raw_fd(), 
                    libc::SOL_SOCKET, 
                    libc::SO_SNDBUF, 
                    len.as_mut_ptr() as *mut libc::c_void, 
                    &mut {std::mem::size_of::<libc::socklen_t>() as libc::socklen_t} 
                )
            };

        if res == 0
        {
            let mut len = unsafe { len.assume_init() };

            if len < MAXLINE
            {
                len = MAXLINE;

                unsafe {
                    libc::setsockopt(
                        stream.get_raw_fd(), 
                        libc::SOL_SOCKET, 
                        libc::SO_SNDBUF, 
                        &len as *const _ as *const libc::c_void, 
                        std::mem::size_of::<libc::socklen_t>() as libc::socklen_t
                    )
                };
            }
        }

        *self.stream.borrow_mut() = stream;

        return Ok(());
    }

    /// An internal function which is called by the syslog or vsyslog.
    /// A glibc implementation RFC3164
    #[cfg(target_os = "linux")]
    fn vsyslog1<S: AsRef<str>>(&self, mut pri: i32, fmt: S)
    {
        // check for invalid bits
        match check_invalid_bits(&mut pri)
        {
            Ok(_) => {},
            Err(_e) => self.vsyslog1(get_internal_log(), fmt.as_ref())
        }

        // check priority against setlogmask
        if self.option.is_logmasked(pri) == true
        {
            return;
        }

        // set default facility if not specified in pri
        if (pri & LOG_FACMASK) == 0
        {
            pri |= self.option.get_logfacility().bits();
        }

        /*let mut hostname_buf = [0u8; MAXHOSTNAMELEN];
        let hostname = 
            match nix::unistd::gethostname(&mut hostname_buf)
            {
                Ok(r) =>
                {
                    match r.to_str()
                    {
                        Ok(r) => r,
                        Err(_e) => NILVALUE
                    }
                },
                Err(_e) => NILVALUE,
            };*/

        // get timedate
        let timedate = Local::now().format("%h %e %T").to_string();

        // get appname
        if self.option.exists_logtag() == false
        {
            match portable::p_getprogname()
            {
                Some(r) => self.option.set_logtag(r),
                None => self.option.set_logtag("unknown")
            }
        }

        let b_progname = self.option.get_logtag();
        let progname = b_progname.as_ref().unwrap();
        

        let msg = fmt.as_ref();
        let msg_final = 
            if msg.ends_with("\n") == true
            {
                truncate(msg)
            }
            else
            {
                msg
            };

        // message based on RFC 3164
        let msg_pri = 
            [
                "<".as_bytes(), pri.to_string().as_bytes(), ">".as_bytes()
            ].concat();

        let msg_header = 
            [ 
                // timedate
                timedate.as_bytes(), 
                // hostname
               // " ".as_bytes(), hostname.as_bytes(), 
            ].concat();

        let mut msg = 
            [
                // appname
                " ".as_bytes(), progname.as_bytes(), 
                // PID
                "[".as_bytes(), portable::get_pid().to_string().as_str().as_bytes(), "]:".as_bytes(),
                // msg
                " ".as_bytes(), /*b"\xEF\xBB\xBF",*/ msg_final.as_bytes()
            ].concat();
        
        drop(b_progname);

        // output to stderr if required
        self.option.send_to_stderr(&mut msg);
        
        
        let fullmsg = [msg_pri.as_slice(), msg_header.as_slice(), msg.as_slice()].concat();

        if self.stream.borrow().is_none() == true
        {
            // open connection
            match self.connectlog()
            {
                Ok(_) => {},
                Err(e) =>
                {
                    self.option.send_to_stderr(unsafe { e.eject_string().as_bytes_mut() } );
                    return;
                }
            }
        }

        // There are two possible scenarios when send may fail:
        // 1. syslog temporary unavailable
        // 2. syslog out of buffer space
        // If we are connected to priv socket then in case of 1 we reopen connection
        //      and retry once.
        // If we are connected to unpriv then in case of 2 repeatedly retrying to send
        //      until syslog socket buffer space will be cleared

        loop
        {
            let mut stream = self.stream.borrow_mut();

            match stream.send(&fullmsg)
            {
                Ok(_) => return,
                Err(err) =>
                {   
                    if let Some(libc::ENOBUFS) = err.raw_os_error()
                    {
                        // scenario 2
                        if stream.is_priv() == true
                        {
                            break;
                        }

                        std::thread::sleep(std::time::Duration::from_micros(1));
                        drop(stream);
                    }
                    else
                    {
                        // scenario 1
                        drop(stream);

                        self.disconnectlog();
                        match self.connectlog()
                        {
                            Ok(_) => {},
                            Err(_e) => break,
                        }

                        // if resend will fail then probably the scn 2 will take place
                    }   
                }
            }
        } // loop


        // If program reached this point then transmission over socket failed.
        // Try to output message to console

        if self.option.is_logstat_flag(LogStat::LOG_CONS)
        {
            let fd = unsafe {
                libc::open(
                    PATH_CONSOLE.as_ptr(), 
                    libc::O_WRONLY | libc::O_NONBLOCK | libc::O_CLOEXEC, 
                    0
                )
            };

            if fd >= 0
            {
                let mut without_pri = [msg_header.as_slice(), msg.as_slice()].concat();
                let mut newline = String::from("\r\n");
                send_to_stderr(fd, without_pri.as_mut_slice(),&mut newline);

                unsafe {libc::close(fd)};
            }
        }
    }

    /// An internal function which is called by the syslog or vsyslog.
    /// A glibc implementation RFC5424
    #[cfg(any(
        target_os = "freebsd",
        target_os = "dragonfly",
        target_os = "openbsd",
        target_os = "netbsd",
        target_os = "macos"
    ))]
    fn vsyslog1<S: AsRef<str>>(&self, mut pri: i32, fmt: S)
    {
        // check for invalid bits
        match check_invalid_bits(&mut pri)
        {
            Ok(_) => {},
            Err(_e) => self.vsyslog1(get_internal_log(), fmt.as_ref())
        }

        // check priority against setlogmask
        if self.option.is_logmasked(pri) == true
        {
            return;
        }

        // set default facility if not specified in pri
        if (pri & LOG_FACMASK) == 0
        {
            pri |= self.option.get_logfacility().bits();
        }

        let mut hostname_buf = [0u8; MAXHOSTNAMELEN];
        let hostname = 
            match nix::unistd::gethostname(&mut hostname_buf)
            {
                Ok(r) =>
                {
                    match r.to_str()
                    {
                        Ok(r) => r,
                        Err(_e) => NILVALUE
                    }
                },
                Err(_e) => NILVALUE,
            };

        // get timedate
        let timedate = 
            Local::now().to_rfc3339_opts(SecondsFormat::Secs, false);

        // get appname
        if self.option.exists_logtag() == false
        {
            match portable::p_getprogname()
            {
                Some(r) => self.option.set_logtag(r),
                None => self.option.set_logtag(NILVALUE)
            }
        }

        let b_progname = self.option.get_logtag();
        let progname = b_progname.as_ref().unwrap();
        

        let msg = fmt.as_ref();
        let msg_final = 
            if msg.ends_with("\n") == true
            {
                truncate(msg)
            }
            else
            {
                msg
            };

        // message based on RFC 5424
        let msg_pri = 
            [
                "<".as_bytes(), pri.to_string().as_bytes(), ">1".as_bytes()
            ].concat();

        let msg_header = 
            [ 
                // timedate
                " ".as_bytes(), timedate.as_bytes(), 
                // hostname
                " ".as_bytes(), hostname.as_bytes(), 
            ].concat();
        
        let mut msg = 
            [
                // appname
                " ".as_bytes(), progname.as_bytes(), 
                // PID
                " ".as_bytes(), portable::get_pid().to_string().as_str().as_bytes(),
                // message ID
                " ".as_bytes(), NILVALUE.as_bytes(), 
                // structured data
                " ".as_bytes(), NILVALUE.as_bytes(), 
                // msg
                " ".as_bytes(), /*b"\xEF\xBB\xBF",*/ msg_final.as_bytes()
            ].concat();
        
        drop(progname);

        // output to stderr if required
        self.option.send_to_stderr(&mut msg);
        
        
        let fullmsg = [msg_pri.as_slice(), msg_header.as_slice(), msg.as_slice()].concat();

        if self.stream.borrow().is_none() == true
        {
            // open connection
            match self.connectlog()
            {
                Ok(_) => {},
                Err(e) =>
                {
                    self.option.send_to_stderr(unsafe { e.eject_string().as_bytes_mut() } );
                    return;
                }
            }
        }

        // There are two possible scenarios when send may fail:
        // 1. syslog temporary unavailable
        // 2. syslog out of buffer space
        // If we are connected to priv socket then in case of 1 we reopen connection
        //      and retry once.
        // If we are connected to unpriv then in case of 2 repeatedly retrying to send
        //      until syslog socket buffer space will be cleared

        loop
        {
            let mut stream = self.stream.borrow_mut();

            match stream.send(&fullmsg)
            {
                Ok(_) => return,
                Err(err) =>
                {   
                    if let Some(libc::ENOBUFS) = err.raw_os_error()
                    {
                        // scenario 2
                        if stream.is_priv() == true
                        {
                            break;
                        }

                        std::thread::sleep(std::time::Duration::from_micros(1));
                        drop(stream);
                    }
                    else
                    {
                        // scenario 1
                        drop(stream);

                        self.disconnectlog();
                        match self.connectlog()
                        {
                            Ok(_) => {},
                            Err(_e) => break,
                        }

                        // if resend will fail then probably the scn 2 will take place
                    }   
                }
            }
        } // loop


        // If program reached this point then transmission over socket failed.
        // Try to output message to console

        if self.option.is_logstat_flag(LogStat::LOG_CONS)
        {
            let fd = unsafe {
                libc::open(
                    PATH_CONSOLE.as_ptr(), 
                    libc::O_WRONLY | libc::O_NONBLOCK | libc::O_CLOEXEC, 
                    0
                )
            };

            if fd >= 0
            {
                let mut without_pri = [msg_header.as_slice(), msg.as_slice()].concat();
                let mut newline = String::from("\r\n");
                send_to_stderr(fd, without_pri.as_mut_slice(),&mut newline);

                unsafe {libc::close(fd)};
            }
        }

    }
}

/// A public implementations
impl Syslog
{
    /// As in a libc, this function initializes the syslog instance. The 
    /// main difference with realization in C is it returns the instance
    /// to program used this crate. This structure implements the [Send] 
    /// and [Sync] so it does not require any additional synchonization.
    ///
    /// # Arguments
    /// 
    /// * `ident` - a identification of the sender. If not set, the crate
    ///             will determine automatically!
    /// * `logstat` - sets up the syslog behaviour. Use [LogStat] 
    /// 
    /// * `facility` - a syslog facility. Use [LogFacility]
    /// 
    /// # Returns
    ///
    /// * A [SyRes] with instance or Err()
    ///
    /// # Example
    /// 
    /// ```
    ///  Syslog::openlog(
    ///        Some("test1"), 
    ///         LogStat::LOG_NDELAY | LogStat::LOG_PID, 
    ///         LogFacility::LOG_DAEMON);
    /// ```
    pub fn openlog(
        ident: Option<&str>, 
        logstat: LogStat, 
        facility: LogFacility) -> SyRes<Self>
    {
        let ret = Self
            {
                lock: AtomicBool::new(false),
                option: CachePadded::new(SyslogOption::new(ident, logstat, facility)),
                stream: RefCell::new(SyslogSocket::none()),
            };

        if logstat.contains(LogStat::LOG_NDELAY) == true
        {
            ret.connectlog()?;
        }

        return Ok(ret);
    }

    /// Sets the logmask to filter out the syslog calls.
    /// 
    /// See macroses [LOG_MASK] and [LOG_UPTO] to generate mask
    ///
    /// # Example
    ///
    /// LOG_MASK!(Priority::LOG_EMERG) | LOG_MASK!(Priority::LOG_ERROR)
    ///
    /// or
    ///
    /// ~(LOG_MASK!(Priority::LOG_INFO))
    /// LOG_UPTO!(Priority::LOG_ERROR)
    pub fn setlogmask(&self, logmask: i32) -> i32
    {
        let backoff = Backoff::new();

        // try lock
        while self.lock.swap(true, Ordering::Acquire) == true
        {
            backoff.snooze();
        }

        // locked

        let pri = self.option.set_logmask(logmask);

        // unlock 
        self.lock.store(false, Ordering::Release);

        return pri;
    }

    /// Similar to libc, closelog() will close the log
    pub fn closelog(&self)
    {
        let backoff = Backoff::new();

        // try lock
        while self.lock.swap(true, Ordering::Acquire) == true
        {
            backoff.snooze();
        }
        
        self.disconnectlog();

        // unlock
        self.lock.store(false, Ordering::Release);
    }

    /// Similar to libc, syslog() sends data to syslog server.
    /// 
    /// # Arguments
    ///
    /// * `pri` - a priority [Priority]
    ///
    /// * `fmt` - a string message. In C exists a functions with
    ///     variable argumets amount. In Rust you should create your
    ///     own macros like format!() or use format!()
    pub fn syslog(&self, pri: Priority, fmt: String)
    {
        self.vsyslog(pri, fmt);
    }

    /// Similar to syslog() and created for the compatability.
    pub fn vsyslog<S: AsRef<str>>(&self, pri: Priority, fmt: S)
    {
        let backoff = Backoff::new();

        // try lock
        while self.lock.swap(true, Ordering::Acquire) == true
        {
            backoff.snooze();
        }

        self.vsyslog1(pri.bits(), fmt);

        // unlock
        self.lock.store(false, Ordering::Release);
    }

    // --- NON STANDART API

    /// This function can be used to update the facility name, for example
    /// after fork().
    /// 
    /// # Arguments
    /// 
    /// * `ident` - a new identity (up to 48 UTF8 chars)
    pub fn change_identity<I: AsRef<str>>(&self, ident: I)
    {
        let backoff = Backoff::new();

        // try lock
        while self.lock.swap(true, Ordering::Acquire) == true
        {
            backoff.snooze();
        }

        self.option.set_logtag(ident);

        // unlock
        self.lock.store(false, Ordering::Release);
    }
}

#[test]
fn test_single_message()
{
    /*use std::sync::Arc;
    use std::thread;
    use std::time::Duration;
    use super::{LOG_MASK};*/

    let log = 
            Syslog::openlog(
                Some("test1"), 
                LogStat::LOG_CONS | LogStat::LOG_NDELAY | LogStat::LOG_PID, 
                LogFacility::LOG_DAEMON);

    assert_eq!(log.is_ok(), true, "{}", log.err().unwrap());

    let log = log.unwrap();

    log.syslog(Priority::LOG_DEBUG, format!("test_set_logmask() проверка BOM"));

    log.closelog();

    return;
}

#[test]
fn test_multithreading()
{
    use std::sync::Arc;
    use std::thread;
    use std::time::{Instant, Duration};

    let log = 
            Syslog::openlog(
                Some("test1"), 
                LogStat::LOG_CONS | LogStat::LOG_NDELAY | LogStat::LOG_PID, 
                LogFacility::LOG_DAEMON);

    assert_eq!(log.is_ok(), true, "{}", log.err().unwrap());

    let log = Arc::new(log.unwrap());
    let c1_log = log.clone();
    let c2_log = log.clone();

    thread::spawn(move|| {
        for i in 0..5
        {
            thread::sleep(Duration::from_nanos(200));
            let now = Instant::now();
            c1_log.syslog(Priority::LOG_DEBUG, format!("a message from thread 1 #{}[]", i));
            let elapsed = now.elapsed();
            println!("t1: {:?}", elapsed);
        }
    });

    thread::spawn(move|| {
        for i in 0..5
        {
            thread::sleep(Duration::from_nanos(201));
            let now = Instant::now();
            c2_log.syslog(Priority::LOG_DEBUG, format!("сообщение от треда 2 №{}ХЪ", i));
            let elapsed = now.elapsed();
            println!("t2: {:?}", elapsed);
        }
    });

    let now = Instant::now();
    log.syslog(Priority::LOG_DEBUG, format!("A message from main, сообщение от главнюка"));
    let elapsed = now.elapsed();
    println!("main: {:?}", elapsed);

    thread::sleep(Duration::from_secs(2));

    log.closelog();

    return;
}
