/*-
* syslog-rs - a syslog client translated from libc to rust
* Copyright (C) 2021  Aleksandr Morozov
* 
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU Lesser General Public
* License as published by the Free Software Foundation; either
* version 3 of the License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
* Lesser General Public License for more details.
* 
* You should have received a copy of the GNU Lesser General Public License
* along with this program; if not, write to the Free Software Foundation,
* Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*/

//! syslog-rs
//! 
//! An implementation of the syslog from glibc/libc like it was designed in
//! in both system libraries. The API is almost compatible with what is in
//! libc/glibc.
//! 
//! Supported GNU/Linux and *BSD (incl OSX, but not guaranteed)
//! 
//! Files:
//! - syslog_sync.rs - contains the thread-safe realization of the syslog (sync).
//!     Thread safe. The Atomic and crossbeam backoff are used for sync.
//! - syslog_async.rs - contains the async realization of the syslog (async)
//!     Thread safe. Tokio mutex are used for sync.
//! - syslog_sync_queue.rs - constains the sync realization, with asynchronious 
//!     processing. The internal sych of crossbeam are used to push data to queue.
//! - unsafe_cell.rs - a file contains a Cell which can be used to share the syslog
//!     instance. See examples.
//! - portable.rs - all system level code which is portable
//! - common.rs - a common items
//! - socket.rs - contains socket realization
//! - async_socket.rs - contains socket realization
//! - error.rs - an error wrapper and mapper
//! 
//! Features: 
//! - feature = "use_async" for asynchronious code (use syslog_rs::sy_async::{Syslog};)
//! - feature = "use_sync" for synchronious code (use syslog_rs::sy_sync::{Syslog};)
//! - feature = "use_sync_queue" for synchronious with async processing (use syslog_rs::sy_async_queue::{Syslog};)
//!
//! All features can be used simultaniously.
//! # Usage
//! 
//! syslog-rs = {version = "0.1", default-features = false, features = ["use_sync"]}
//! 
//! By default, `use_sync` and `use_async` are enabled.

#[macro_use] extern crate bitflags;
#[cfg(any(feature = "use_sync", feature = "use_sync_queue"))]
extern crate crossbeam;
extern crate chrono;
extern crate nix;
extern crate libc;
#[macro_use] extern crate lazy_static;

#[cfg(feature = "use_async")]
extern crate async_recursion;
#[cfg(feature = "use_async")]
extern crate tokio;

#[cfg(feature = "use_sync")]
pub mod syslog_sync;
#[cfg(feature = "use_async")]
pub mod syslog_async;
#[cfg(feature = "use_sync_queue")]
pub mod syslog_sync_queue;

#[cfg(any(feature = "use_sync", feature = "use_sync_queue"))]
pub mod socket;
#[cfg(feature = "use_async")]
pub mod async_socket;

pub mod unsafe_cell;
#[macro_use] pub mod common;
#[macro_use] pub mod error;
pub mod portable;

#[cfg(feature = "use_sync")]
pub use syslog_sync as sy_sync;
#[cfg(feature = "use_async")]
pub use syslog_async as sy_async;
#[cfg(feature = "use_sync_queue")]
pub use syslog_sync_queue as sy_sync_queue;

pub use common::*;
pub use unsafe_cell::*;
