
#[cfg(feature = "use_async")]
use tokio::net::UnixDatagram;

#[cfg(not(feature = "use_async"))]
use std::os::unix::net::UnixDatagram;

use std::{
    net::Shutdown,
    os::unix::{
        prelude::AsRawFd, 
        io::RawFd
    }
};

use nix::errno::Errno;

use crate::{map_error, throw_error};

use super::common::*;
use super::error::{SyRes, SyslogError, SyslogErrCode};

/// A private enum with type of the socket
pub(crate) enum SyslogSocket
{
    /// Not initialized
    None,

    /// Unprivileged socket used
    UnPriv(UnixDatagram),

    /// Privileged socket used
    Priv(UnixDatagram),

    /// A compatibility socket used
    OldLog(UnixDatagram),
}

unsafe impl Sync for SyslogSocket{}
unsafe impl Send for SyslogSocket{}

impl SyslogSocket
{
    /// Creates a default instance
    pub(crate) fn none() -> Self
    {
        return Self::None;
    }

    /// Returns the rawfd without passing the ownership ower the
    /// the unixsocket. Will panic if called on enum None.
    pub(crate) fn get_raw_fd(&self) -> RawFd
    {
        match *self
        {
            Self::None => panic!("Assertion: wrong use of get_raw_fd()"),
            Self::UnPriv(ref s) | 
            Self::Priv(ref s) | 
            Self::OldLog(ref s) =>
            {
                return s.as_raw_fd();
            }
        }
    }

    /// Performs the connection to the syslog's server socket.
    /// Picks the socket automatically.
    pub(crate) fn connect() -> SyRes<Self>
    {
        let sock = 
            UnixDatagram::unbound().map_err(|e|
                map_error!("unbounded unix datagram create failed: {}", e)
            )?;

        if let Ok(_) = sock.connect(PATH_LOG_PRIV)
        {
            return Ok(Self::Priv(sock));
        }
        else if let Ok(_) = sock.connect(PATH_LOG)
        {
            return Ok(Self::UnPriv(sock));
        }
        else if let Ok(_) = sock.connect(PATH_OLDLOG)
        {
            return Ok(Self::OldLog(sock));
        }
        else if let Ok(_) = sock.connect(PATH_OSX)
        {
            return Ok(Self::Priv(sock));
        }
        else
        {
            // failed to open socket
            throw_error!("failed to open connection to syslog server: {} ", 
                        Errno::last());
        };
    }

    /// Writes to the socket. Will panic if instance is enum None.
    #[cfg(not(feature = "use_async"))]
    pub(crate) fn send(&mut self, msg: &[u8]) -> std::io::Result<usize>
    {
        match *self
        {
            Self::None => panic!("Assertion: wrong use of get_raw_fd()"),
            Self::UnPriv(ref mut s) | 
            Self::Priv(ref mut s) | 
            Self::OldLog(ref mut s) =>
            {
                return s.send(msg);
            }
        } 
    }

    #[cfg(feature = "use_async")]
    pub(crate) async fn send(&mut self, msg: &[u8]) -> std::io::Result<usize>
    {
        match *self
        {
            Self::None => panic!("Assertion: wrong use of get_raw_fd()"),
            Self::UnPriv(ref mut s) | 
            Self::Priv(ref mut s) | 
            Self::OldLog(ref mut s) =>
            {
                return s.send(msg).await;
            }
        } 
    }

    /// Shutdowns the socket. Will panic if instance is enum None.
    pub(crate) fn shutdown(&mut self)
    {
        match *self
        {
            Self::None => panic!("Assertion: wrong use of get_raw_fd()"),
            Self::UnPriv(ref mut s) | 
            Self::Priv(ref mut s) | 
            Self::OldLog(ref mut s) =>
            {
                let _ = s.shutdown(Shutdown::Both);
            }
        }
    }

    /// Returns true is current instance of enum is None.
    pub(crate) fn is_none(&self) -> bool
    {
        match self
        {
            Self::None => return true,
            Self::UnPriv(_) | 
            Self::Priv(_) | 
            Self::OldLog(_) => return false,
        }
    }

    /// Returns true if stream was connected to privileged socket.
    pub(crate) fn is_priv(&self) -> bool
    {
        match self
        {
            Self::None => return false,
            Self::UnPriv(_) => return false,
            Self::Priv(_) => return true,
            Self::OldLog(_) => return false,
        }
    }
}
