/*-
* syslog-rs - a syslog client translated from libc to rust
* Copyright (C) 2021  Aleksandr Morozov
* 
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU Lesser General Public
* License as published by the Free Software Foundation; either
* version 3 of the License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
* Lesser General Public License for more details.
* 
* You should have received a copy of the GNU Lesser General Public License
* along with this program; if not, write to the Free Software Foundation,
* Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*/

//! syslog-rs
//! 
//! An implementation of the syslog from glibc/libc like it was designed in
//! in both system libraries. The API is almost compatible with what is in
//! libc/glibc.
//! 
//! Supported GNU/Linux and *BSD (incl OSX, but not guaranteed)
//! 
//! Files:
//! - syslog_sync.rs - contains the thread-safe realization of the syslog (sync)
//! - syslog_async.rs - contains the async realization of the syslog (async)
//! - syslog_sync_queue.rs - constains the sync realization, with asynchronious 
//!     processing.
//! - portable.rs - all system level code which is portable
//! - common.rs - a common items
//! - socket.rs - contains socket realization
//! - error.rs - an error wrapper and mapper
//! 
//! Features: 
//! - feature = "use_async" for asynchronious code
//! - feature = "use_sync" for synchronious code
//! - feature = "use_sync_queue" for synchronious with async processing

#[macro_use] extern crate bitflags;
#[cfg(not(feature = "use_async"))]
extern crate crossbeam;
extern crate chrono;
extern crate nix;
extern crate libc;
#[macro_use] extern crate lazy_static;

#[cfg(feature = "use_async")]
extern crate async_recursion;
#[cfg(feature = "use_async")]
extern crate tokio;

#[cfg(feature = "use_sync")]
pub mod syslog_sync;
#[cfg(feature = "use_async")]
pub mod syslog_async;
#[cfg(feature = "use_sync_queue")]
pub mod syslog_sync_queue;

pub mod socket;
#[macro_use] pub mod common;
#[macro_use] pub mod error;
pub mod portable;

#[cfg(all(feature = "use_sync", feature = "use_async"))]
compile_error!("feature \"use_sync\" and feature \"use_async\" cannot be enabled at the same time");
#[cfg(all(feature = "use_sync", feature = "use_sync_queue"))]
compile_error!("feature \"use_sync\" and feature \"use_sync_queue\" cannot be enabled at the same time");
#[cfg(all(feature = "use_async", feature = "use_sync_queue"))]
compile_error!("feature \"use_async\" and feature \"use_sync_queue\" cannot be enabled at the same time");

#[cfg(feature = "use_sync")]
pub use syslog_sync::*;
#[cfg(feature = "use_async")]
pub use syslog_async::*;
#[cfg(feature = "use_sync_queue")]
pub use syslog_sync_queue::*;

pub use common::*;
