# syslog-rs

An implementation of the syslog from glibc/libc like it was designed in
in both system libraries. The API is almost compatible with what is in
libc/glibc.

Available features:
- feature = "use_async" for asynchronious code
- feature = "use_sync" for synchronious code
- feature = "use_sync_queue" for synchronious with async processing

The `use_sync` is acting like the libc's/glibc's functions syslog(), openlog()...

The `use_sync_queue` has the same API as libc/glibc but is different in some ways. Also, it spawns a worker thread which sends messages from the queue to syslog.

The `use_async` is async realization of the `use_sync`. Untested, probably requires further dev.

## Usage:

syslog-rs = {version = "0.1", default-features = false, features = ["use_sync"]}

## Supports

- GNU/Linux RFC3164 (UTF-8 by default)
- *BSD and OSX RFC5424 (BOM UTF-8 by default)

## Example

```rust
#[macro_use] extern crate lazy_static;
#[macro_use] extern crate syslog_rs;

use std::sync::Arc;
use std::thread;
use std::time::{Instant, Duration};

use syslog_rs::{Syslog, LogStat, LogFacility, Priority};


lazy_static! {
    static ref SYSLOG: Arc<Syslog> = 
        Arc::new(
            Syslog::openlog(
                Some("example"), 
                LogStat::LOG_CONS | LogStat::LOG_NDELAY | LogStat::LOG_PID, 
                LogFacility::LOG_DAEMON
            ).unwrap()
        );
}

macro_rules! logdebug 
{
    ($($arg:tt)*) => (
        SYSLOG.syslog(Priority::LOG_DEBUG, format!($($arg)*))
    )
}



fn main()
{
    logdebug!("test message!");

    thread::sleep(Duration::from_micros(10));

    return;
}
```

## Benchmarking

The test spawns 2 threads and one main thread. All 3 threads are sending messages to syslog. The time measurment in the tables are approximations.

Results of the tests in syslog_*.rs files in Debug mode:

| use_sync         | use_sync_queue |   use_async  |
|----------        |----------------|--------------|
|main: 100.778µs   | t1: 6.681µs    |main: 87.323µs|
|t1: 57.026µs       |t2: 14.692µs   |t2: 74.494µs   |
|t2: 111.32µs       |t1: 728ns      |t1: 284.415µs  |
|t1: 108.239µs      |main: 1.967µs  |t1: 57.522µs   |
|t2: 101.264µs      |t1: 2.222µs    |t2: 100.814µs  |
|t1: 269.168µs      |t2: 758ns      |t2: 57.693µs   |
|t2: 95.979µs       |t1: 632ns      |t1: 37.736µs   |
|t1: 38.424µs       |t2: 1.097µs    |t1: 71.579µs   |
|t1: 28.495µs       |t1: 487ns      |t2: 308.357µs  |
|t2: 62.484µs       |t2: 1.381µs    |t2: 85.555µs   |
|t2: 46.874µs       |t2: 2.317µs    |t1: 39.13µs   |
|  ------           | -------       | ---------     |
|main: 80.361µs     |t2: 9.708µs    |main: 116.662µs|
|t1: 71.085µs       |main: 2.987µs  |t2: 95.859µs   |
|t1: 41.886µs       |t1: 36.827µs   |t1: 161.768µs  |
|t1: 26.027µs       |t2: 7.537µs    |t1: 214.533µs  |
|t1: 38.499µs       |t1: 1.313µs    |t2: 411.774µs  |
|t1: 25.682µs       |t2: 801ns      |t2: 78.55µs    |
|t2: 51.021µs       |t1: 499ns      |t1: 137.938µs  |
|t2: 51.115µs       |t1: 7.228µs    |t2: 131.843µs  |
|t2: 37.879µs       |t2: 1.194µs    |t2: 273.557µs  |
|t2: 27.949µs       |t1: 935ns      |t1: 1.810786ms |
|t2: 50.807µs       |t2: 628ns      |t1: 1.66199ms  |


