/*-
* syslog-rs - a syslog client translated from libc to rust
* Copyright (C) 2021  Aleksandr Morozov
* 
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU Lesser General Public
* License as published by the Free Software Foundation; either
* version 3 of the License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
* Lesser General Public License for more details.
* 
* You should have received a copy of the GNU Lesser General Public License
* along with this program; if not, write to the Free Software Foundation,
* Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*/

use std::ffi::CStr;
use std::path::Path;

#[cfg(any(
    target_os = "freebsd",
    target_os = "dragonfly",
    target_os = "openbsd",
    target_os = "netbsd",
    target_os = "macos"
))]
#[link(name = "c")]
extern "C" {
    fn getprogname() -> *const libc::c_char;
}

#[cfg(target_os = "linux")]
#[link(name = "c")]
extern "C" {
    pub static mut program_invocation_name : *mut libc::c_char ;
}

/// Reutns the current process name, if available
pub(crate) fn p_getprogname() -> Option<String>
{
    #[cfg(any(
        target_os = "freebsd",
        target_os = "dragonfly",
        target_os = "openbsd",
        target_os = "netbsd",
        target_os = "macos"
    ))]
    let pn = getprogname();

    #[cfg(target_os = "linux")]
    let pn = unsafe{ program_invocation_name };

    let temp = unsafe {CStr::from_ptr(pn)};

    match temp.to_str()
    {
        Ok(r) => 
        {
            #[cfg(target_os = "linux")]
            {
                let path = Path::new(r);
                match path.file_name()
                {
                    Some(r) => return Some(r.to_string_lossy().into()),
                    None => return None,
                }
            }

            #[cfg(any(
                target_os = "freebsd",
                target_os = "dragonfly",
                target_os = "openbsd",
                target_os = "netbsd",
                target_os = "macos"
            ))]
            return Some(r.to_string());
        },
        Err(_) => return None,
    }
   
}


#[cfg(target_os = "netbsd")]
#[link(name = "c")]
extern "C" {
    fn _lwp_self() -> libc::c_uint;
}

#[cfg(any(
    target_os = "freebsd",
    target_os = "dragonfly",
    target_os = "openbsd",
))]
#[link(name = "pthread")]
extern "C" {
    fn pthread_getthreadid_np() -> c_int;
}

/// Returns pid of current process. Not thread id!
pub fn get_pid() -> u32
{
    return std::process::id();
}

#[test]
fn test_get_procname()
{
    println!("Processname is: {:?}", p_getprogname());
}
