use std::sync::Arc;

use structopt::StructOpt;

use syndicate::actor::*;
use syndicate::relay;
use syndicate::sturdy;
use syndicate::value::Value;

use tokio::net::TcpStream;

#[derive(Clone, Debug, StructOpt)]
pub struct Config {
    #[structopt(short = "a", default_value = "1")]
    action_count: u32,

    #[structopt(short = "b", default_value = "0")]
    bytes_padding: usize,

    #[structopt(short = "d", default_value = "b4b303726566b10973796e646963617465b584b210a6480df5306611ddd0d3882b546e197784")]
    dataspace: String,
}

#[inline]
fn says(who: AnyValue, what: AnyValue) -> AnyValue {
    let mut r = Value::simple_record("Says", 2);
    r.fields_vec_mut().push(who);
    r.fields_vec_mut().push(what);
    r.finish().wrap()
}

#[tokio::main]
async fn main() -> Result<(), Box<dyn std::error::Error>> {
    syndicate::convenient_logging()?;
    Actor::new().boot(syndicate::name!("producer"), |t| {
        let facet = t.facet.clone();
        let boot_account = Arc::clone(t.account());
        Ok(t.linked_task(tracing::Span::current(), async move {
            let config = Config::from_args();
            let sturdyref = sturdy::SturdyRef::from_hex(&config.dataspace)?;
            let (i, o) = TcpStream::connect("127.0.0.1:8001").await?.into_split();
            facet.activate(boot_account, |t| {
                relay::connect_stream(t, i, o, sturdyref, (), move |_state, t, ds| {
                    let padding: AnyValue = Value::ByteString(vec![0; config.bytes_padding]).wrap();
                    let action_count = config.action_count;
                    let account = Account::new(syndicate::name!("account"));
                    t.linked_task(syndicate::name!("sender"), async move {
                        loop {
                            account.ensure_clear_funds().await;
                            let mut events: PendingEventQueue = Vec::new();
                            for _ in 0..action_count {
                                let ds = Arc::clone(&ds);
                                let padding = padding.clone();
                                events.push(Box::new(move |t| t.with_entity(
                                    &ds.underlying,
                                    |t, e| e.message(t, says(Value::from("producer").wrap(), padding)))));
                            }
                            external_events(&ds.underlying.mailbox, &account, events)?;
                        }
                    });
                    Ok(None)
                });
                Ok(())
            })
        }))
    }).await??;
    Ok(())
}
