use quick_protobuf::MessageRead;
use quick_protobuf::serialize_into_vec;
use solana_program::instruction::AccountMeta;
use solana_program::{account_info::AccountInfo, program::invoke, program_error::ProgramError};
use switchboard_protos::protos::instruction::SwitchboardInstruction;
use switchboard_protos::protos::instruction::mod_SwitchboardInstruction::GetAggregateInstruction;
use switchboard_protos::protos::instruction::mod_SwitchboardInstruction::OneOfinstruction::get_aggregate_instruction;

pub use switchboard_protos::protos::aggregator_state::RoundResult;

pub fn deserialize_from_slice<'a, M: MessageRead<'a>>(bytes: &'a [u8]) -> Result<M, ProgramError> {
    quick_protobuf::deserialize_from_slice(bytes)
        .map_err(|_| ProgramError::InvalidInstructionData)
}

pub fn invoke_feed<'a>(
    switchboard: &'a AccountInfo<'a>,
    switchboard_feed: &'a AccountInfo<'a>,
    callback: &'a AccountInfo<'a>,
    callback_signer: &'a AccountInfo<'a>,
    other_accounts: &'a [AccountInfo<'a>],
) -> Result<(), ProgramError> {
    let switchboard_instruction = SwitchboardInstruction {
        instruction: get_aggregate_instruction(GetAggregateInstruction {}),
    };

    let mut account_metas = vec![
        AccountMeta::new(*switchboard_feed.key, false),
        AccountMeta::new(*callback.key, false),
        AccountMeta::new(*callback_signer.key, true),
    ];
    let mut other_account_metas: Vec<AccountMeta> = other_accounts
        .to_vec()
        .iter_mut()
        .map(|account_info| AccountMeta::new(*account_info.key, false))
        .collect();
    account_metas.append(&mut other_account_metas);
    let solana_instruction = solana_program::instruction::Instruction {
        program_id: *switchboard.key,
        data: serialize_into_vec(&switchboard_instruction).unwrap(),
        accounts: account_metas,
    };
    let mut accounts = vec![
        switchboard.clone(),
        switchboard_feed.clone(),
        callback.clone(),
        callback_signer.clone(),
    ];
    accounts.append(&mut other_accounts.to_vec());
    invoke(&solana_instruction, &accounts)?;
    Ok(())
}
