# swift-bridge [![Actions Status](https://github.com/chinedufn/swift-bridge/workflows/test/badge.svg)](https://github.com/chinedufn/swift-bridge/actions) [![docs](https://docs.rs/swift-bridge/badge.svg)](https://docs.rs/swift-bridge)

> `swift-bridge` facilitates Rust and Swift interop.

## Book

You can find information about using Rust and Swift together in [The `swift-bridge` book](https://chinedufn.github.io/swift-bridge).

## Quick Start

The `swift-bridge` repository contains [example applications](examples) that you use to quickly try out the library,
or as a starting point for your own `Swift` + `Rust` based application.

For example, here's how to run the [`ios-rust-analyzer`](examples/ios-rust-analyzer) example project locally.

```
git clone https://github.com/chinedufn/swift-bridge
cd swift-bridge/examples/ios-rust-analyzer

open IosRustAnalyzer/IosRustAnalyzer.xcodeproj
# Click the "Run" button at the top left of Xcode
```

## Quick Peek

Share Swift and Rust types and functions between the two languages.

```rust
// Rust

// You write the type signatures of your FFI boundary in Rust,
// which `swift-bridge` then uses to generate the FFI layer.
#[swift_bridge::bridge]
mod ffi {
    // Create structs that both Swift and Rust can use.
    #[swift_bridge(swift_repr = "struct")]
    struct Comparison {
        summary: Option<String>,
    }

    // Export Rust types and functions for Swift to use.
    extern "Rust" {
        type ARustStack;

        #[swift_bridge(init)]
        fn new() -> ARustStack;

        fn push(&mut self, val: String);
        fn pop(&mut self) -> Option<String>;
    }

    // Import Swift types and functions for Rust to use.
    extern "Swift" {
        type ASwiftGraph;

        fn compare_graphs(g1: &ASwiftGraph, g2: &ASwiftGraph) -> Comparison;
    }
}

struct ARustStack {
    stack: Vec<String>
}
impl ARustStack {
    fn new() -> ARustStack { /* ... */ }
    fn push(&mut self, val: String) { /* ... */ }
    fn pop(&mut self) -> Option<String> { /* ... */ }
}
```

```swift
// Swift

let stack = ARustStack()
stack.push("Hello, hello.")
let hello = stack.pop()!

class ASwiftGraph {
    // ...
}

func compare_graphs(g1: &ASwiftGraph, g2: &ASwiftGraph) -> Comparison {
    // ...
    return Comparison(summary: "Things went well.")
}
```

## Installation

```toml
# In your Cargo.toml

[build-dependencies]
swift-bridge-build = "0.1"

[dependencies]
swift-bridge = "0.1"
```

## Built-In Types

In addition to allowing you to share your own custom types between Rust and Swift,
`swift_bridge` comes with support for a number of Rust and Swift standard library types.

| name in Rust                                                    | name in Swift                                                    | notes               |
| ---                                                             | ---                                                              | ---                 |
| u8, i8, u16, i16... etc                                         | UInt8, Int8, UInt16, Int16 ... etc                               |                     |
| bool                                                            | Bool                                                             |                     |
| String, &String, &mut String                                    | RustString, RustStringRef, RustStringRefMut                      |                     |
| &str                                                            | RustStr                                                          |                     |
| Vec\<T>                                                         | RustVec\<T>                                                      |                     |
| SwiftArray\<T>                                                  | Array\<T>                                                        | Not yet implemented |
| &[T]                                                            | UnsafeBufferPointer\<T>                                          |                     |
| &mut [T]                                                        | UnsafeMutableBufferPointer\<T>                                   | Not yet implemented |
| SwiftString                                                     | String                                                           |                     |
| Box<T>                                                          |                                                                  | Not yet implemented |
| [T; N]                                                          |                                                                  | Not yet implemented |
| *const T                                                        | UnsafePointer\<T>                                                |                     |
| *mut T                                                          | UnsafeMutablePointer\<T>                                         |                     |
| Option\<T>                                                      | Optional\<T>                                                     |                     |
| Result\<T>                                                      |                                                                  | Not yet implemented |
| Have a Rust standard library type in mind?<br /> Open an issue! |                                                                  |                     |
|                                                                 | Have a Swift standard library type in mind?<br /> Open an issue! |                     |

## To Test

To run the test suite.

```sh
# Clone the repository
git clone git@github.com:chinedufn/swift-bridge.git
cd swift-bridge

# Run tests
cargo test --all && ./test-integration.sh
```

## Phases

#### Phase 1 (Current Phase): Make it Possible

Bridging Rust and Swift is fairly unexplored territory, so it will take some experimentation in order to
figure out the right APIs and code generation.

During this phase we'll focus on adding support for more types, patterns and and common use cases
that we discover.

We won't be overly focused on what the best names or arguments or structure during this phase.

#### Phase 2: Make it Ergonomic

This phase will be focused on making `swift-bridge` feel really good to use.

During this phase we will:

- Simplify our APIs and make them consistent.

- Improve our error messages.

- Improve the information and examples in the book.

#### Phase 3: Make it Stable

This phase is about getting `swift-bridge` to version `1.0.0`.

We'll take inventory of all of our public APIs and aim to remove as much we
can without impacting the libraries usability.

---

The `0.1.x` versions will not follow semver.

We will maintain semver from `0.2` and onwards.

---

#### License

<sup>
Licensed under either of <a href="LICENSE-APACHE">Apache License, Version
2.0</a> or <a href="LICENSE-MIT">MIT license</a> at your option.
</sup>

<br>

<sub>
Unless you explicitly state otherwise, any contribution intentionally submitted
for inclusion in this project by you, as defined in the Apache-2.0 license,
shall be dual licensed as above, without any additional terms or conditions.
</sub>
