/*
	Copyright (c) 2021 Lucy <lucy@absolucy.moe>

	This Source Code Form is subject to the terms of the Mozilla Public
	License, v. 2.0. If a copy of the MPL was not distributed with this
	file, You can obtain one at http://mozilla.org/MPL/2.0/.
*/

use crate::{error::Error, state::State};
use chacha20::cipher::{StreamCipher, StreamCipherSeek};

/// A 64-bit ID, generated by the [sled] database.
/// Contains logic for converting to an English identifier and back.
#[derive(Clone, Copy)]
pub struct Id([u8; 8]);

impl Id {
	pub async fn decode(state: &State, s: &str) -> Result<Self, Error> {
		let mut id = [0_u8; 8];
		// Decode the id to bytes.
		let offset = mnemonic::decode(s, &mut id as &mut [u8])?;
		// Ensure the length is between 1 and 8 bytes.
		if !(1..=8).contains(&offset) {
			return Err(Error::InvalidId);
		}
		// Rotate any padding zeroes to the front.
		id.rotate_right(8 - offset);
		// Decrypt the id.
		let mut chacha = state.chacha.lock().await;
		chacha.apply_keystream(&mut id[8 - offset..]);
		// Seek the cipher back to 0.
		chacha.seek(0);
		// Return the decoded id
		Ok(Self(id))
	}

	pub async fn encode(&self, state: &State) -> String {
		let mut id = [0_u8; 8];
		// Find the first non-zero byte, that's where we'll start.
		let first_nonzero = self.0.iter().position(|byte| *byte != 0).unwrap_or(0);
		// This asset is for optimization purposes.
		assert_eq!(id[first_nonzero..].len(), self.0[first_nonzero..].len());
		// Copy the non-zero ID bytes to our array.
		id[first_nonzero..].copy_from_slice(&self.0[first_nonzero..]);
		// Encrypt the id.
		let mut chacha = state.chacha.lock().await;
		chacha.apply_keystream(&mut id[first_nonzero..]);
		// Seek the cipher back to 0.
		chacha.seek(0);
		// Encode the id to a string.
		mnemonic::to_string(&id[first_nonzero..])
	}
}

impl From<i64> for Id {
	fn from(u: i64) -> Self {
		Self(u.to_be_bytes())
	}
}

impl From<Id> for i64 {
	fn from(id: Id) -> Self {
		i64::from_be_bytes(id.0)
	}
}
