use crate::jsx::JSXText;
use num_bigint::BigInt as BigIntValue;
use serde::{Deserialize, Serialize};
use std::{
    fmt::{self, Display, Formatter},
    hash::{Hash, Hasher},
    mem,
};
use swc_atoms::{js_word, JsWord};
use swc_common::{ast_node, util::take::Take, EqIgnoreSpan, Span, DUMMY_SP};

#[ast_node]
#[derive(Eq, Hash, EqIgnoreSpan)]
#[cfg_attr(feature = "arbitrary", derive(arbitrary::Arbitrary))]
pub enum Lit {
    #[tag("StringLiteral")]
    Str(Str),

    #[tag("BooleanLiteral")]
    Bool(Bool),

    #[tag("NullLiteral")]
    Null(Null),

    #[tag("NumericLiteral")]
    Num(Number),

    #[tag("BigIntLiteral")]
    BigInt(BigInt),

    #[tag("RegExpLiteral")]
    Regex(Regex),

    #[tag("JSXText")]
    JSXText(JSXText),
}

#[ast_node("BigIntLiteral")]
#[derive(Eq, Hash, EqIgnoreSpan)]
pub struct BigInt {
    pub span: Span,
    #[cfg_attr(feature = "rkyv", with(EncodeBigInt))]
    pub value: BigIntValue,
}

#[cfg(feature = "rkyv")]
#[derive(Debug, Clone, Copy)]
pub struct EncodeBigInt;

#[cfg(feature = "rkyv")]
impl rkyv::with::ArchiveWith<BigIntValue> for EncodeBigInt {
    type Archived = rkyv::Archived<String>;

    type Resolver = rkyv::Resolver<String>;

    unsafe fn resolve_with(
        field: &BigIntValue,
        pos: usize,
        resolver: Self::Resolver,
        out: *mut Self::Archived,
    ) {
        use rkyv::Archive;

        let s = field.to_string();
        s.resolve(pos, resolver, out);
    }
}

#[cfg(feature = "rkyv")]
impl<S> rkyv::with::SerializeWith<BigIntValue, S> for EncodeBigInt
where
    S: ?Sized + rkyv::ser::Serializer,
{
    fn serialize_with(field: &BigIntValue, serializer: &mut S) -> Result<Self::Resolver, S::Error> {
        let field = field.to_string();
        rkyv::string::ArchivedString::serialize_from_str(&field, serializer)
    }
}

#[cfg(feature = "rkyv")]
impl<D> rkyv::with::DeserializeWith<rkyv::Archived<String>, BigIntValue, D> for EncodeBigInt
where
    D: ?Sized + rkyv::Fallible,
{
    fn deserialize_with(
        field: &rkyv::Archived<String>,
        deserializer: &mut D,
    ) -> Result<BigIntValue, D::Error> {
        use rkyv::Deserialize;

        let s: String = field.deserialize(deserializer)?;

        Ok(s.parse().unwrap())
    }
}

#[cfg(feature = "arbitrary")]
#[cfg_attr(docsrs, doc(cfg(feature = "arbitrary")))]
impl<'a> arbitrary::Arbitrary<'a> for BigInt {
    fn arbitrary(u: &mut arbitrary::Unstructured<'_>) -> arbitrary::Result<Self> {
        let span = u.arbitrary()?;
        let value = u.arbitrary::<usize>()?.into();

        Ok(Self { span, value })
    }
}

/// A string literal.
///
/// # Note
///
/// You have to use [StrKind::Synthesized] if you modify the `value` of [Str].
/// This behavior is for preserving the original source code.
///
/// In other words, `swc_ecma_codegen` tries to preserve escapes or unicode
/// characters in the original source code and you can opt-out of this by using
/// [StrKind::Synthesized].
#[ast_node("StringLiteral")]
#[derive(Eq, Hash, EqIgnoreSpan)]
pub struct Str {
    pub span: Span,

    #[cfg_attr(feature = "rkyv", with(crate::EncodeJsWord))]
    pub value: JsWord,

    /// This includes line escape.
    #[serde(default)]
    pub has_escape: bool,

    #[serde(default)]
    pub kind: StrKind,
}

impl Take for Str {
    fn dummy() -> Self {
        Str {
            span: DUMMY_SP,
            value: js_word!(""),
            has_escape: Default::default(),
            kind: Default::default(),
        }
    }
}

/// THis enum determines how string literal should be printed.
#[derive(Debug, Clone, Copy, PartialEq, Eq, Hash, Serialize, Deserialize)]
#[serde(tag = "type")]
#[cfg_attr(
    feature = "rkyv",
    derive(rkyv::Archive, rkyv::Serialize, rkyv::Deserialize)
)]
pub enum StrKind {
    /// Span of string points to original source code, and codegen should use
    /// it.
    //
    /// **Note**: Giving wrong value to this field will result in invalid
    /// codegen.
    #[serde(rename = "normal")]
    Normal {
        /// Does span of this string literal contains quote?
        ///
        /// True for string literals generated by parser, false for string
        /// literals generated by various passes.
        #[serde(rename = "containsQuote")]
        #[cfg_attr(feature = "rkyv", omit_bounds)]
        contains_quote: bool,
    },
    /// If the span of string does not point a string literal, mainly because
    /// this string is synthesized, this variant should be used.
    #[serde(rename = "synthesized")]
    Synthesized,
}

/// Always returns true as this is not a data of a string literal.
impl EqIgnoreSpan for StrKind {
    fn eq_ignore_span(&self, _: &Self) -> bool {
        true
    }
}

impl Default for StrKind {
    fn default() -> Self {
        Self::Synthesized
    }
}

#[cfg(feature = "arbitrary")]
#[cfg_attr(docsrs, doc(cfg(feature = "arbitrary")))]
impl<'a> arbitrary::Arbitrary<'a> for Str {
    fn arbitrary(u: &mut arbitrary::Unstructured<'_>) -> arbitrary::Result<Self> {
        let span = u.arbitrary()?;
        let value = u.arbitrary::<String>()?.into();

        Ok(Self {
            span,
            value,
            has_escape: false,
            kind: Default::default(),
        })
    }
}

impl Str {
    #[inline]
    pub fn is_empty(&self) -> bool {
        self.value.is_empty()
    }
}

#[ast_node("BooleanLiteral")]
#[derive(Copy, Eq, Hash, EqIgnoreSpan)]
#[cfg_attr(feature = "arbitrary", derive(arbitrary::Arbitrary))]
pub struct Bool {
    pub span: Span,
    pub value: bool,
}

impl Take for Bool {
    fn dummy() -> Self {
        Bool {
            span: DUMMY_SP,
            value: false,
        }
    }
}

#[ast_node("NullLiteral")]
#[derive(Copy, Eq, Hash, EqIgnoreSpan)]
#[cfg_attr(feature = "arbitrary", derive(arbitrary::Arbitrary))]
pub struct Null {
    pub span: Span,
}

impl Take for Null {
    fn dummy() -> Self {
        Null { span: DUMMY_SP }
    }
}

#[ast_node("RegExpLiteral")]
#[derive(Eq, Hash, EqIgnoreSpan)]
pub struct Regex {
    pub span: Span,

    #[serde(rename = "pattern")]
    #[cfg_attr(feature = "rkyv", with(crate::EncodeJsWord))]
    pub exp: JsWord,

    #[serde(default)]
    #[cfg_attr(feature = "rkyv", with(crate::EncodeJsWord))]
    pub flags: JsWord,
}

#[cfg(feature = "arbitrary")]
#[cfg_attr(docsrs, doc(cfg(feature = "arbitrary")))]
impl<'a> arbitrary::Arbitrary<'a> for Regex {
    fn arbitrary(u: &mut arbitrary::Unstructured<'_>) -> arbitrary::Result<Self> {
        let span = u.arbitrary()?;
        let exp = u.arbitrary::<String>()?.into();
        let flags = "".into(); // TODO

        Ok(Self { span, exp, flags })
    }
}

#[ast_node("NumericLiteral")]
#[derive(Copy, EqIgnoreSpan)]
#[cfg_attr(feature = "arbitrary", derive(arbitrary::Arbitrary))]
pub struct Number {
    pub span: Span,
    /// **Note**: This should not be `NaN`. Use [crate::Ident] to represent NaN.
    ///
    /// If you store `NaN` in this field, a hash map will behave strangely.
    #[use_eq]
    pub value: f64,
}

impl Eq for Number {}

impl Hash for Number {
    fn hash<H: Hasher>(&self, state: &mut H) {
        fn integer_decode(val: f64) -> (u64, i16, i8) {
            let bits: u64 = unsafe { mem::transmute(val) };
            let sign: i8 = if bits >> 63 == 0 { 1 } else { -1 };
            let mut exponent: i16 = ((bits >> 52) & 0x7ff) as i16;
            let mantissa = if exponent == 0 {
                (bits & 0xfffffffffffff) << 1
            } else {
                (bits & 0xfffffffffffff) | 0x10000000000000
            };

            exponent -= 1023 + 52;
            (mantissa, exponent, sign)
        }

        self.span.hash(state);
        integer_decode(self.value).hash(state);
    }
}

impl Display for Number {
    fn fmt(&self, f: &mut Formatter<'_>) -> fmt::Result {
        if self.value.is_infinite() {
            if self.value.is_sign_positive() {
                Display::fmt("Infinity", f)
            } else {
                Display::fmt("-Infinity", f)
            }
        } else {
            Display::fmt(&self.value, f)
        }
    }
}
