// A grep-like tool for separated values files.
//
// Copyright (C) 2017-2021  Tassilo Horn <tsdh@gnu.org>
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the Free
// Software Foundation; either version 3 of the License, or (at your option)
// any later version.
//
// This program is distributed in the hope that it will be useful, but WITHOUT
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
// FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
// more details.
//
// You should have received a copy of the GNU General Public License along with
// this program; if not, write to the Free Software Foundation, Inc., 51
// Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA

use clap::{App, Arg, ArgMatches};
use svgrep::{build_match_exp, line_iter, svgrep_lines, Config, MatchCharCfg};

fn main() {
    let opts = parse_command_line();
    let config = build_config(&opts);

    let lines = line_iter(opts.value_of(OPT_FILE));
    svgrep_lines(lines, config);
}

fn build_config(opts: &ArgMatches) -> Config {
    let match_char_cfg = MatchCharCfg {
        cell_select_char: String::from(opts.value_of(OPT_SELECT_CHAR).unwrap_or("@")),
        match_conj_char: String::from(opts.value_of(OPT_CONJ_CHAR).unwrap_or("&")),
        matches_char: String::from(opts.value_of(OPT_MATCHES_CHAR).unwrap_or("=")),
    };

    Config {
        separator: String::from(opts.value_of(OPT_SEPARATOR).unwrap_or(";")),
        trim: opts.is_present(OPT_TRIM),
        match_exps: opts
            .values_of(OPT_MATCH)
            .unwrap_or_default()
            .map(|match_val| build_match_exp(match_val, &match_char_cfg))
            .collect(),
    }
}

const OPT_FILE: &str = "FILE";
const OPT_SEPARATOR: &str = "separator";
const OPT_MATCH: &str = "match";
const OPT_CONJ_CHAR: &str = "conj-char";
const OPT_SELECT_CHAR: &str = "cell-select-char";
const OPT_MATCHES_CHAR: &str = "matches-char";
const OPT_TRIM: &str = "trim";
const VERSION: Option<&str> = option_env!("CARGO_PKG_VERSION");

fn parse_command_line<'a>() -> ArgMatches<'a> {
    App::new("svgrep -- Separated Values Grep")
        .version(VERSION.unwrap_or("<version unknown>"))
        .about("Greps and extracts cells of CSV/TSV/*SV files")
        .author("Tassilo Horn <tsdh@gnu.org>")
        .arg(
            Arg::with_name(OPT_FILE)
                .help("The separated values file. If none is given, reads from stdin.")
                .required(false),
        )
        .arg(
            Arg::with_name(OPT_SEPARATOR)
                .short("s")
                .long(OPT_SEPARATOR)
                .takes_value(true)
                .value_name("char")
                .help("Sets the separator to be used (default: ';')"),
        )
        .arg(
            Arg::with_name(OPT_MATCH)
                .short("m")
                .long(OPT_MATCH)
                .takes_value(true)
                .multiple(true)
                .help(
                    format!(
                        "{}\n{}\n{}\n{}\n{}\n{}\n{}",
                        "Sets the match-and-select expression.\n",
                        "Syntax:\n<col>=<regex>(&<col>=<regex>)+@<disp_cols>",
                        "<col> is a natural number or * meaning any column.",
                        "<regex> is a regex matched against the cells at column <col>.",
                        "<disp_cols> is a comma-separated list of columns to display (defaul: all).",
                        "\n--match '1=foo&2=bar' acts as logical AND wheras multiple expressions like",
                        "--match '1=foo' '2=bar' act as a logical OR."
                    ).as_str(),
                ),
        )
        .arg(Arg::with_name(OPT_MATCHES_CHAR)
             .short("=")
             .long(OPT_MATCHES_CHAR)
             .takes_value(true)
             .value_name("char")
             .help(format!("{}\n{}",
                           "Separates a <col> from the <regex> in --match expressions.",
                           "(default: =).").as_str()))
        .arg(Arg::with_name(OPT_CONJ_CHAR)
             .short("&")
             .long(OPT_CONJ_CHAR)
             .takes_value(true)
             .value_name("char")
             .help(format!("{}\n{}",
                           "Separates multiple <col>=<regex> pairs in --match expressions",
                           "to form a conjunction (default: &).").as_str()))
        .arg(Arg::with_name(OPT_SELECT_CHAR)
             .short("@")
             .long(OPT_SELECT_CHAR)
             .takes_value(true)
             .value_name("char")
             .help(format!("{}\n{}",
                           "Separates the <col>=<regex> pairs in --match expressions from",
                           "the column display selection (default: @).").as_str()))
        .arg(Arg::with_name(OPT_TRIM)
             .short("t")
             .long(OPT_TRIM)
             .help("Trim the cell contents when printing."))
        .get_matches()
}
