// A grep-like tool for separated values files.
//
// Copyright (C) 2017-2021  Tassilo Horn <tsdh@gnu.org>
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the Free
// Software Foundation; either version 3 of the License, or (at your option)
// any later version.
//
// This program is distributed in the hope that it will be useful, but WITHOUT
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
// FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
// more details.
//
// You should have received a copy of the GNU General Public License along with
// this program; if not, write to the Free Software Foundation, Inc., 51
// Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA

use lazy_static::lazy_static;
use regex::Regex;
use std::collections::HashMap;
use std::fs::File;
use std::io::{self, BufRead, BufReader, Lines};
use std::process::exit;

pub struct CSVRow {
    cells: Vec<String>,
}

pub enum CellSelect {
    All,
    Some(Vec<usize>),
}

pub struct MatchExp {
    pub rxs: Vec<Regex>,
    pub cell_rxs: HashMap<usize, Regex>,
    pub sel: CellSelect,
}

pub struct Config {
    pub separator: String,
    pub trim: bool,
    pub match_exps: Vec<MatchExp>,
}

pub struct MatchCharCfg {
    pub cell_select_char: String,
    pub match_conj_char: String,
    pub matches_char: String,
}

impl MatchExp {
    fn new() -> MatchExp {
        MatchExp {
            rxs: vec![],
            cell_rxs: HashMap::new(),
            sel: CellSelect::All,
        }
    }

    fn match_and_select(&self, row: &CSVRow, config: &Config) {
        let mut row_matches = self.rxs.is_empty() && self.cell_rxs.is_empty();

        row_matches = row_matches
            || self.cell_rxs.iter().all(|(cell_idx, rx)| {
                let cell = row.get_cell(*cell_idx);
                cell.is_some() && rx.is_match(cell.unwrap())
            });
        row_matches = row_matches
            && self
                .rxs
                .iter()
                .all(|rx| row.cells.iter().any(|cell| rx.is_match(cell)));

        if row_matches {
            row.print(&self.sel, config);
        }
    }
}

impl CSVRow {
    fn from_line(line: String, sep: &str) -> CSVRow {
        CSVRow {
            cells: line.split(sep).map(String::from).collect(),
        }
    }

    fn get_cell(&self, idx: usize) -> Option<&str> {
        if idx >= self.cells.len() {
            None
        } else {
            Some(self.cells[idx].as_str())
        }
    }

    fn print(&self, cols: &CellSelect, config: &Config) {
        match cols {
            CellSelect::All => {
                for (i, cell) in self.cells.iter().enumerate() {
                    print!("({}) {} ", i, maybe_trim(cell, config.trim));
                }
            }
            CellSelect::Some(ref cols) => {
                for i in cols {
                    if i >= &self.cells.len() {
                        print!("<no col {}>", i);
                    } else {
                        print!(
                            "({}) {}",
                            i,
                            maybe_trim(self.cells[*i].as_str(), config.trim)
                        );
                    }
                    print!("{} ", config.separator);
                }
            }
        }
        println!();
    }
}

fn maybe_trim(cell: &str, trim: bool) -> &str {
    if trim {
        cell.trim()
    } else {
        cell
    }
}

pub fn line_iter(file_name: Option<&str>) -> Lines<Box<dyn BufRead>> {
    let reader: Box<dyn BufRead> = match file_name {
        None => Box::new(BufReader::new(io::stdin())),
        Some(filename) => Box::new(BufReader::new(
            File::open(filename).unwrap_or_else(|_| panic!("No file {}", filename)),
        )),
    };
    reader.lines()
}

pub fn svgrep_lines(lines: Lines<Box<dyn BufRead>>, config: Config) {
    let all_match = &vec![MatchExp::new()];
    let match_exps = if config.match_exps.is_empty() {
        all_match
    } else {
        &config.match_exps
    };

    for row in lines.map(|l| CSVRow::from_line(l.unwrap(), &config.separator)) {
        for match_exp in match_exps {
            match_exp.match_and_select(&row, &config);
        }
    }
}

fn error(msg: &str) {
    eprintln!("Error: {}", msg);
    exit(1);
}

lazy_static! {
    static ref NUMBER_RX: Regex = Regex::new(r"^\d+.*$").expect("Invalid Regex in the code!");
    static ref ASTERISK_RX: Regex =
        Regex::new([r"^", regex::escape("*").as_str(), "$"].join("").as_ref())
            .expect("Invalid Regex in the code!");
}

fn build_rxs(
    m: Option<regex::Match>,
    match_char_cfg: &MatchCharCfg,
) -> (Vec<Regex>, HashMap<usize, Regex>) {
    match m {
        None => (vec![], HashMap::new()),
        Some(m) => {
            let match_clauses: Vec<&str> =
                m.as_str().split(&match_char_cfg.match_conj_char).collect();
            let mut v = Vec::new();
            let mut hm = HashMap::new();

            for clause in match_clauses {
                let col_and_rx: Vec<&str> = clause.split(&match_char_cfg.matches_char).collect();
                if NUMBER_RX.is_match(col_and_rx[0]) {
                    hm.insert(
                        col_and_rx[0]
                            .parse::<usize>()
                            .expect("Invalid match column!"),
                        Regex::new(col_and_rx[1]).expect("Invalid regex!"),
                    );
                } else if ASTERISK_RX.is_match(col_and_rx[0]) {
                    v.push(Regex::new(col_and_rx[1]).expect("Invalid regex!"));
                } else {
                    error(format!("'{}' is no valid column spec!", col_and_rx[0]).as_str());
                }
            }

            (v, hm)
        }
    }
}

fn build_cell_select(s: Option<regex::Match>) -> CellSelect {
    match s {
        None => CellSelect::All,
        Some(v) => CellSelect::Some(
            v.as_str()
                .split(',')
                .map(|is| is.parse::<usize>().expect("Invalid index in select!"))
                .collect(),
        ),
    }
}

pub fn build_match_exp(match_val: &str, match_char_cfg: &MatchCharCfg) -> MatchExp {
    let rx = Regex::new(
        [
            r"^([^",
            regex::escape(&match_char_cfg.cell_select_char).as_ref(),
            "]+)?(?:",
            regex::escape(&match_char_cfg.cell_select_char).as_ref(),
            r"(\d+(,\d+)*))?$",
        ]
        .join("")
        .as_ref(),
    )
    .unwrap();

    let captures = rx.captures(match_val).expect("Invalid --match expression!");

    let (rxs, cell_rxs) = build_rxs(captures.get(1), match_char_cfg);
    MatchExp {
        rxs,
        cell_rxs,
        sel: build_cell_select(captures.get(2)),
    }
}
