use std::env;
use std::fs;
use std::io::Write;
use std::process::Command;
use std::{
  ffi::{OsStr, OsString},
  time::Duration
};

use windows_service::{
  define_windows_service,
  service::{
    ServiceAccess, ServiceControl, ServiceControlAccept, ServiceExitCode,
    ServiceState, ServiceStatus, ServiceType
  },
  service_control_handler::{self, ServiceControlHandlerResult},
  service_dispatcher,
  service_manager::{ServiceManager, ServiceManagerAccess}
};

use figment::Figment;
use figment_winreg::RegistryProvider;

use serde::Deserialize;

use log::{debug, error, info};

use winreg::{enums::*, RegKey};

use qargparser as arg;

use crate::err::Error;


const SERVICE_TYPE: ServiceType = ServiceType::OWN_PROCESS;
const SERVICE_STARTPENDING_TIME: Duration = Duration::from_secs(30);
const SERVICE_STOPPENDING_TIME: Duration = Duration::from_secs(30);


#[derive(Default)]
pub struct Context {
  svcname: String
}


#[derive(Deserialize, Debug)]
pub struct Config {
  /// List of strings representing command and arguments to run during
  /// initialization.
  #[serde(rename = "InitHandler")]
  inithandler: Option<Vec<String>>,

  /// List of strings representing command and arguments to run.
  #[serde(rename = "Handler")]
  handler: Option<Vec<String>>,

  /// List of strings representing command and arguments to run during
  /// termination.
  #[serde(rename = "TermHandler")]
  termhandler: Option<Vec<String>>,

  /// Log file used to capture output of handlers
  #[serde(rename = "CaptureLog")]
  capturelog: Option<String>,

  /// List of services to start once handler has run and files have been
  /// copied.
  #[serde(rename = "StartServices")]
  services: Option<Vec<String>>,

  #[serde(rename = "LogLevel")]
  loglevel: Option<String>
}


/// Kick off service dispatch loop.
pub fn run() -> Result<(), Error> {
  // Wait for a debugger to attach, and then break
  #[cfg(feature = "dbgtools-win")]
  dbgtools_win::debugger::wait_for_then_break();

  let ctx = parse()?;

  service_dispatcher::start(ctx.svcname, ffi_service_main)?;

  Ok(())
}


// Generate the windows service boilerplate.  The boilerplate contains the
// low-level service entry function (ffi_service_main) that parses incoming
// service arguments into Vec<OsString> and passes them to user defined service
// entry (my_service_main).
define_windows_service!(ffi_service_main, my_service_main);


fn my_service_main(_arguments: Vec<OsString>) {
  let _ = inner_main();
}


/// Service entry function which is called on background thread by the system
/// with service parameters.  There is no stdout or stderr at this point so
/// make sure to configure the log.
fn inner_main() -> Result<(), Error> {
  let ctx = match parse() {
    Ok(ctx) => ctx,
    Err(_) => panic!("Unable to parse command line")
  };

  let pth = format!(
    r"SYSTEM\CurrentControlSet\Services\{}\Parameters",
    ctx.svcname
  );

  let fig =
    Figment::new().merge(RegistryProvider::new(HKEY_LOCAL_MACHINE, pth));
  let config: Config = fig.extract()?;

  // Set up logging level from registry parameter.
  // Defaults to error level
  let lf = if let Some(ll) = config.loglevel {
    match ll.as_ref() {
      "off" => log::LevelFilter::Off,
      "error" => log::LevelFilter::Error,
      "warn" => log::LevelFilter::Warn,
      "info" => log::LevelFilter::Info,
      "debug" => log::LevelFilter::Debug,
      "trace" => log::LevelFilter::Trace,
      _ => log::LevelFilter::Error
    }
  } else {
    log::LevelFilter::Error
  };

  // Initailize log
  eventlog::init(&ctx.svcname, log::Level::Trace).unwrap();
  log::set_max_level(lf);

  debug!("Setting up service");

  // Define system service event handler that will be receiving service events.
  // Don't currently handle stop events.
  let event_handler = move |control_event| -> ServiceControlHandlerResult {
    match control_event {
      ServiceControl::Interrogate => {
        debug!("svc signal recieved: interrogate");
        // Notifies a service to report its current status information to the
        // service control manager. Always return NoError even if not
        // implemented.
        ServiceControlHandlerResult::NoError
      }
      ServiceControl::Stop => {
        debug!("svc signal recieved: stop");
        ServiceControlHandlerResult::NoError
      }
      ServiceControl::Continue => {
        debug!("svc signal recieved: continue");
        ServiceControlHandlerResult::NotImplemented
      }
      ServiceControl::Pause => {
        debug!("svc signal recieved: pause");
        ServiceControlHandlerResult::NotImplemented
      }
      _ => {
        debug!("svc signal recieved: other");
        ServiceControlHandlerResult::NotImplemented
      }
    }
  };

  // Switch working directory if set.
  if let Some(wd) = get_service_param(&ctx.svcname, "WorkDir") {
    env::set_current_dir(wd)?;
  }

  // Register system service event handler.  (The returned status handle
  // should be used to report service status changes to the system).  And
  // report that we're in the process of starting up.
  let status_handle =
    service_control_handler::register(&ctx.svcname, event_handler)?;

  status_handle.set_service_status(ServiceStatus {
    service_type: SERVICE_TYPE,
    current_state: ServiceState::StartPending,
    controls_accepted: ServiceControlAccept::empty(),
    exit_code: ServiceExitCode::Win32(0),
    checkpoint: 0,
    wait_hint: SERVICE_STARTPENDING_TIME,
    process_id: None
  })?;


  //
  // Clear capture log if used
  //
  if let Some(ref caplog) = config.capturelog {
    match std::fs::File::create(caplog) {
      Ok(_) => {
        debug!("Cleared capture log");
      }
      Err(e) => {
        error!("Unable to clear capture log; {}", e);
      }
    }
  }


  //
  // If an initialization handler has been set, then run it while in
  // StartPending state.
  //
  if let Some(handler) = config.inithandler {
    run_handler(
      &handler[0],
      &handler[1..],
      config.capturelog.clone(),
      "InitHandler"
    );
  }


  status_handle.set_service_status(ServiceStatus {
    service_type: SERVICE_TYPE,
    current_state: ServiceState::Running,
    controls_accepted: ServiceControlAccept::STOP,
    exit_code: ServiceExitCode::Win32(0),
    checkpoint: 0,
    wait_hint: Duration::default(),
    process_id: None
  })?;


  //
  // If a handler has been set, then run it.
  //
  if let Some(handler) = config.handler {
    run_handler(
      &handler[0],
      &handler[1..],
      config.capturelog.clone(),
      "Handler"
    );
  }


  //
  // Start services
  //
  if let Some(services) = config.services {
    for svc in services {
      let manager_access = ServiceManagerAccess::CONNECT;
      let service_manager =
        ServiceManager::local_computer(None::<&str>, manager_access)?;
      let service_access = ServiceAccess::QUERY_STATUS | ServiceAccess::START;
      let service = service_manager.open_service(&svc, service_access)?;

      // Make sure service is stopped before trying to delete it
      let service_status = service.query_status()?;
      if service_status.current_state == ServiceState::Stopped {
        info!("Requesting service '{}' to start", svc);

        // Need to figure out how to initialize this from the service.
        let args: Vec<OsString> = Vec::new();
        match service.start(&args) {
          Ok(_) => {
            info!("Started service '{}'", svc);
          }
          Err(e) => {
            error!("Unable to start service '{}'; {}", svc, e);
          }
        }
      }
    }
  }


  status_handle
    .set_service_status(ServiceStatus {
      service_type: SERVICE_TYPE,
      current_state: ServiceState::StopPending,
      controls_accepted: ServiceControlAccept::empty(),
      exit_code: ServiceExitCode::Win32(0),
      checkpoint: 0,
      wait_hint: SERVICE_STOPPENDING_TIME,
      process_id: None
    })
    .unwrap();


  //
  // If an initialization handler has been set, then run it while in
  // StartPending state.
  //
  if let Some(handler) = config.termhandler {
    run_handler(
      &handler[0],
      &handler[1..],
      config.capturelog.clone(),
      "TermHandler"
    );
  }


  status_handle
    .set_service_status(ServiceStatus {
      service_type: SERVICE_TYPE,
      current_state: ServiceState::Stopped,
      controls_accepted: ServiceControlAccept::empty(),
      exit_code: ServiceExitCode::Win32(0),
      checkpoint: 0,
      wait_hint: Duration::default(),
      process_id: None
    })
    .unwrap();


  debug!("service terminated");

  Ok(())
}


pub fn parse() -> Result<Context, Error> {
  let ctx = Context {
    ..Default::default()
  };

  let mut prsr = arg::Parser::from_env(ctx);

  prsr.add(
    arg::Builder::new()
      .name("subcmd")
      .required(true)
      .help(&["Subcommand.  Will be set to run-service."])
      .nargs(arg::Nargs::Count(1), &["CMD"])
      .build(|_spec, _ctx: &mut Context, _args| {})
  )?;

  prsr.add(
    arg::Builder::new()
      .name("name")
      .required(true)
      .help(&["Service name."])
      .nargs(arg::Nargs::Count(1), &["NAME"])
      .build(|_spec, ctx: &mut Context, args| {
        ctx.svcname = args[0].clone();
      })
  )?;

  prsr.parse()?;

  Ok(prsr.into_ctx())
}


/// Load a service Parameter from the registry.
pub fn get_service_param(service_name: &str, key: &str) -> Option<String> {
  let hklm = RegKey::predef(HKEY_LOCAL_MACHINE);
  let services = match hklm.open_subkey("SYSTEM\\CurrentControlSet\\Services")
  {
    Ok(k) => k,
    Err(_) => return None
  };
  let asrv = match services.open_subkey(service_name) {
    Ok(k) => k,
    Err(_) => return None
  };
  let params = match asrv.open_subkey("Parameters") {
    Ok(k) => k,
    Err(_) => return None
  };

  match params.get_value::<String, &str>(key) {
    Ok(v) => Some(v),
    Err(_) => None
  }
}

fn run_handler<E, A>(exec: E, args: A, caplog: Option<String>, title: &str)
where
  E: AsRef<OsStr>,
  A: IntoIterator + Copy,
  A::Item: Into<OsString>,
  <A as IntoIterator>::Item: AsRef<OsStr>
{
  //
  // If a capture log was requested then attempt to open it.
  //
  let f = if let Some(caplog) = caplog {
    match fs::OpenOptions::new()
      .write(true)
      .append(true)
      .open(&caplog)
    {
      Ok(mut f) => {
        let _ = writeln!(f, "Running {}:", title);

        Some(f)
      }
      Err(e) => {
        error!("Unable to open {} for writing; {}", caplog, e);
        None
      }
    }
  } else {
    None
  };


  //
  // Run command, and capture its output
  //
  let mut cmd = Command::new(&exec);
  cmd.args(args);
  match cmd.output() {
    Ok(output) => {
      //
      // If a capture log was requested, then dump stdout and stderr to that
      // file.
      //
      if let Some(mut f) = f {
        let _ = writeln!(f, "--- stdout ---");
        f.write_all(&output.stdout).unwrap();
        let _ = writeln!(f, "\n");

        let _ = writeln!(f, "--- stderr ---");
        f.write_all(&output.stderr).unwrap();
        let _ = writeln!(f, "\n");
      }

      if output.status.success() {
        info!("{} run successfully", title);
      } else {
        error!("{} failed; {:?} terminated with error", title, cmd);
      }
    }
    Err(e) => {
      error!("Unable to run {:?}; {}", cmd, e);
    }
  }
}

// vim: set ft=rust et sw=2 ts=2 sts=2 cinoptions=2 tw=79 :
