use crate::class_dictionary::{ClassDictionary, ClassDictionaryError};
use crate::color::{ColorName, CssColor};
use crate::{Classes, ConfigLookup, SunbeamConfig};
use syn::parse::{Parse, ParseStream};
use syn::LitStr;

/// When validating the classes in a proc macro we don't have access to our SunbeamConfig
/// (unless we made our proc macro non-deterministic and added an environment variable that can
/// be used to lookup a SunbeamConfig file.. which we don't want to do), so we
/// can't validate whether or not things like colors or screen sizes are defined in the config.
/// Instead, when we run into things that need to be looked up in config we just assume that
/// they're there. Then later during the end user's build script they'll use their Sunbeam config
/// to do the full validation.
/// This AlwaysSucceedsConfigLookup is just used during procedural macro parsing.
struct AlwaysSucceedsConfigLookup {
    color: CssColor,
}
impl ConfigLookup for AlwaysSucceedsConfigLookup {
    fn get_color(&self, _color_name: &ColorName) -> Option<&CssColor> {
        Some(&self.color)
    }
}

impl Classes {
    /// Parse classes from a string.
    pub fn parse_str(
        classes: &str,
        config: &SunbeamConfig,
    ) -> Result<Self, Vec<ClassDictionaryError>> {
        Self::impl_parse_str(&classes, config)
    }
}

impl Parse for Classes {
    fn parse(input: ParseStream) -> syn::Result<Self> {
        let classes_lit_str: LitStr = input.parse()?;
        let classes: String = classes_lit_str.value();

        let config_lookup = AlwaysSucceedsConfigLookup {
            color: CssColor::new("some-color".to_string()).unwrap(),
        };

        Self::impl_parse_str(&classes, &config_lookup).map_err(|errors| {
            let mut errors = errors.into_iter();

            let mut combined = syn::Error::new(classes_lit_str.span(), errors.next().unwrap());

            while let Some(err) = errors.next() {
                combined.combine(syn::Error::new(classes_lit_str.span(), err));
            }

            combined
        })
    }
}

impl Classes {
    fn impl_parse_str(
        classes: &str,
        config_lookup: &dyn ConfigLookup,
    ) -> Result<Classes, Vec<ClassDictionaryError>> {
        let classes_split = classes.split(" ");

        let class_dict = ClassDictionary::new();

        let mut classes = vec![];
        let mut errors = vec![];

        for class in classes_split {
            match class_dict.get_class_definition(class, config_lookup) {
                Ok(class) => {
                    classes.push(class);
                }
                Err(err) => {
                    errors.push(err);
                }
            };
        }

        if errors.len() > 0 {
            return Err(errors);
        }

        Ok(Classes { classes })
    }
}
