use crate::{Classes, RetrievedClassDefinition};
use std::collections::HashSet;

impl Classes {
    /// Write the classes to a String that can be written to a CSS file.
    pub fn to_css_file_contents(&self) -> String {
        let classes: HashSet<&RetrievedClassDefinition> = self.classes.iter().collect();
        let mut classes: Vec<&RetrievedClassDefinition> = classes.into_iter().collect();
        classes.sort_by(|a, b| {
            let a_hover = a.modifiers.hover();
            let b_hover = b.modifiers.hover();

            if a_hover || b_hover {
                return a_hover.cmp(&b_hover);
            }

            a.css_class_definition.cmp(&b.css_class_definition)
        });

        let mut class_file_contents = "".to_string();

        for class in classes {
            class_file_contents += &class.css_class_definition;
            class_file_contents += "\n";
        }

        class_file_contents.trim().to_string()
    }
}

#[cfg(test)]
mod tests {
    use super::*;
    use crate::{Modifiers, RetrievedClassDefinition};

    /// Verify that we sort the classes in our generated CSS file.
    #[test]
    fn write_css_sorted() {
        let classes = Classes {
            classes: vec![
                RetrievedClassDefinition::new(
                    "".to_string(),
                    r#".mb10 {
    margin-bottom: 10px;
}"#
                    .to_string(),
                    Modifiers::default(),
                ),
                RetrievedClassDefinition::new(
                    "".to_string(),
                    r#".mt5 {
    margin-top: 5px;
}"#
                    .to_string(),
                    Modifiers::default(),
                ),
                RetrievedClassDefinition::new(
                    "".to_string(),
                    r#".mb2 {
    margin-bottom: 2px;
}"#
                    .to_string(),
                    Modifiers::default(),
                ),
            ],
        };

        let contents = classes.to_css_file_contents();

        assert_eq!(
            contents,
            r#"
.mb10 {
    margin-bottom: 10px;
}
.mb2 {
    margin-bottom: 2px;
}
.mt5 {
    margin-top: 5px;
}
"#
            .trim()
        );
    }

    /// Verify that we do not include duplicate copies of a class.
    #[test]
    fn does_not_duplicate() {
        let classes = Classes {
            classes: vec![
                RetrievedClassDefinition::new(
                    "".to_string(),
                    r#".mb10 {
    margin-bottom: 10px;
}"#
                    .to_string(),
                    Modifiers::default(),
                ),
                RetrievedClassDefinition::new(
                    "".to_string(),
                    r#".mb10 {
    margin-bottom: 10px;
}"#
                    .to_string(),
                    Modifiers::default(),
                ),
            ],
        };

        let contents = classes.to_css_file_contents();

        assert_eq!(
            contents,
            r#"
.mb10 {
    margin-bottom: 10px;
}
"#
            .trim()
        );
    }

    /// Verify all hover: classes come last in the generated CSS, since you'll typically want your
    /// hover styles to take precedence over other styles.
    #[test]
    fn hover_comes_before_visited() {
        let mut mod1 = Modifiers::default();
        mod1.set_hover_enabled().unwrap();
        let mod1 = (5, mod1);

        let mod2 = Modifiers::default();
        let mod2 = (10, mod2);

        for (a, b) in vec![(mod1, mod2), (mod2, mod1)] {
            let classes = Classes {
                classes: vec![
                    RetrievedClassDefinition::new(
                        "".to_string(),
                        format!(
                            r#".mb{size} {{
    margin-bottom: {size}px;
}}"#,
                            size = a.0
                        ),
                        a.1,
                    ),
                    RetrievedClassDefinition::new(
                        "".to_string(),
                        format!(
                            r#".mb{size} {{
    margin-bottom: {size}px;
}}"#,
                            size = b.0
                        ),
                        b.1,
                    ),
                ],
            };

            let contents = classes.to_css_file_contents();
            assert!(
                contents.find("margin-bottom: 5px").unwrap()
                    > contents.find("margin-bottom: 10px").unwrap()
            )
        }
    }
}
