use crate::SunbeamConfig;

/// The name of a color, such as the `gray-10` in `text-color-gray-10`.
#[derive(Debug, PartialEq, Eq, PartialOrd, Ord, Hash, serde::Serialize, serde::Deserialize)]
pub struct ColorName(String);

/// An error with a color name.
#[derive(Debug, thiserror::Error)]
pub enum ColorNameError {
    //
}

impl ColorName {
    /// Create a new ColorName.
    pub fn new(name: String) -> Result<ColorName, ColorNameError> {
        Self::validate_color_name(&name)?;
        let name = ColorName(name);
        Ok(name)
    }

    /// Get the name of the color.
    // TODO: We can remove the Result when we implement a custom deserialize for ColorName.
    pub fn get(&self) -> Result<&str, ColorNameError> {
        Self::validate_color_name(&self.0)?;
        Ok(&self.0)
    }

    /// # Example
    ///
    /// red-100 -> text-color-red-100
    pub fn to_text_color_class_name(&self) -> String {
        format!("text-color-{}", self.0)
    }

    /// # Example
    ///
    /// red-100 -> .text-color-red-100 { color: ... }
    pub fn to_text_color_class_body(
        &self,
        config: &SunbeamConfig,
    ) -> Result<String, ColorNameError> {
        // TODO: Errors instead of unwraps

        let color = config.colors.get(self).unwrap();

        let class = format!(r#"color: {};"#, color.get().unwrap());

        Ok(class)
    }

    fn validate_color_name(_color_name: &str) -> Result<(), ColorNameError> {
        // TODO: Validation
        Ok(())
    }
}
