use crate::class_dictionary::{
    ClassDictionary, ClassDictionaryError, ClassSegmentNode, ClassSegmentOrCssTemplate,
};
use crate::color::{ColorName, CssColor};
use crate::{ConfigLookup, Modifiers, Percent, RetrievedClassDefinition};
use std::cell::RefCell;
use std::rc::Rc;
#[derive(Debug)]
enum ClassData {
    Pixels(u32),
    Percent(Percent),
    Color(CssColor),
}

// Arbitrarily chosen number of suggestions to show when you enter an invalid class name.
const MAX_SUGGESTIONS: usize = 4;

impl ClassDictionary {
    pub fn get_class_definition(
        &self,
        class: &str,
        config_lookup: &dyn ConfigLookup,
    ) -> Result<RetrievedClassDefinition, ClassDictionaryError> {
        let mut modifiers = Modifiers::default();

        let class_name = class.replace(":", "\\:");

        let splits = class.split(":");
        let mut last_split = "";
        for split in splits {
            if split == "hover" {
                modifiers.set_hover_enabled().unwrap();
            } else if split == "visited" {
                modifiers.set_visited_enabled().unwrap();
            } else if split.starts_with("gteq") {
                let number = split.trim_start_matches("gteq");

                modifiers
                    .set_min_width_gteq(number.parse::<u32>().unwrap())
                    .unwrap();
            }

            last_split = split;
        }

        let class_without_modifiers = last_split;

        let mut class_data = vec![];
        let mut class_name_template = "".to_string();

        let node = self.root.clone();
        let class_body = self.get_class_body(
            node,
            class_without_modifiers,
            class_without_modifiers,
            &mut class_name_template,
            config_lookup,
            &mut class_data,
        )?;

        let (maybe_media_open, indentation, maybe_media_close) =
            if let Some(gteq) = modifiers.min_width_gteq() {
                (
                    format!("@media (min-width: {}px) {{\n", gteq),
                    "    ",
                    "\n}",
                )
            } else {
                ("".to_string(), "", "")
            };

        let class_name = modifiers.add_modifiers_to_class_name(&class_name);

        let css_class_definition = format!(
            r#"{maybe_media_open}{indentation}.{class_name} {{
{indentation}    {class_body}
{indentation}}}{maybe_media_close}"#,
            class_name = class_name,
            class_body = class_body,
            indentation = indentation,
            maybe_media_open = maybe_media_open,
            maybe_media_close = maybe_media_close
        );

        Ok(RetrievedClassDefinition {
            class_name_template,
            css_class_definition,
            modifiers,
        })
    }

    fn get_class_body(
        &self,
        node: Rc<RefCell<ClassSegmentNode>>,
        class_full_name: &str,
        class_substr: &str,
        class_name_template: &mut String,
        config_lookup: &dyn ConfigLookup,
        class_data: &mut Vec<ClassData>,
    ) -> Result<String, ClassDictionaryError> {
        let node_clone = node.clone();
        let node = node.borrow();

        for (_segment_name, segment) in node.children.iter() {
            match &segment.borrow().node {
                ClassSegmentOrCssTemplate::SpecialSegment(segment_name) => {
                    if segment_name == "{pixels}" || segment_name == "{percent}" {
                        if !class_substr.chars().next().unwrap().is_numeric() {
                            continue;
                        }

                        let mut numbers = "".to_string();
                        'chars: for char in class_substr.chars() {
                            if char.is_numeric() {
                                numbers.push(char);
                            } else {
                                break 'chars;
                            }
                        }

                        let next_idx = numbers.len();
                        let pixels = numbers.parse::<u32>().unwrap();

                        let remaining = &class_substr[next_idx..];

                        if segment_name == "{pixels}" {
                            // Skip if this is really a different type of number.
                            // For example, percentages always end in "pc".
                            let is_percentage = remaining.len() == 2 && remaining.starts_with("pc");
                            if is_percentage {
                                continue;
                            }

                            class_data.push(ClassData::Pixels(pixels));
                        } else if segment_name == "{percent}" {
                            // Skip if this is really a different type of number,
                            // since percentages end in "pc".
                            let is_not_percentage =
                                remaining.len() < 2 || !remaining.starts_with("pc");
                            if is_not_percentage {
                                continue;
                            }

                            class_data.push(ClassData::Percent(Percent::new(pixels).unwrap()));
                        } else {
                            unreachable!();
                        }

                        *class_name_template += segment_name;
                        return self.get_class_body(
                            segment.clone(),
                            class_full_name,
                            remaining,
                            class_name_template,
                            config_lookup,
                            class_data,
                        );
                    } else if segment_name == "{color}" {
                        // Because color nodes can have arbitrary names, we don't allow them to
                        // have siblings. Otherwise we would always be able to match a color node
                        // even if the user really meant to provide a different node.
                        // For example.. Say we had the definitions:
                        // some-class-foo
                        // some-class-{pixels}
                        // We'd now need logic to guarantee that we tried "foo" before we tried
                        // "{pixels}". And we also would need users to know that the "-foo" suffix
                        // even exists and not name any of their colors "foo".
                        // TODO: Add a test that iterates over the class definition trie and
                        //  asserts that none of the `{color}` nodes have siblings.
                        assert_eq!(node.children.len(), 1);

                        let color = config_lookup
                            .get_color(&ColorName::new(class_substr.to_string()).unwrap());

                        if color.is_none() {
                            todo!("Return an error that the color isn't defined. Add a test for this.");
                        }

                        let color = color.unwrap();

                        // Color segments always come last in the class name, so after this will
                        // come the CSS template section.
                        // TODO: Add a test that iterates over the class definition trie and
                        //  asserts that we never have any nodes after the `{color}` node .
                        let class_substr = "";

                        class_data.push(ClassData::Color(color.clone()));

                        *class_name_template += segment_name;
                        return self.get_class_body(
                            segment.clone(),
                            class_full_name,
                            class_substr,
                            class_name_template,
                            config_lookup,
                            class_data,
                        );
                    } else {
                        unreachable!()
                    }
                }
                ClassSegmentOrCssTemplate::Char(char) => {
                    let char = char.to_string();
                    if class_substr.starts_with(&char) {
                        *class_name_template += &char;
                        return self.get_class_body(
                            segment.clone(),
                            class_full_name,
                            &class_substr[1..],
                            class_name_template,
                            config_lookup,
                            class_data,
                        );
                    }
                }
                ClassSegmentOrCssTemplate::CssTemplate(template) => {
                    let mut class_body = template.to_string();

                    for data in class_data {
                        match data {
                            ClassData::Pixels(pixels) => {
                                class_body = class_body
                                    .replace("{pixels}", &pixels.to_string())
                                    .to_string();
                            }
                            ClassData::Percent(percent) => {
                                class_body = class_body
                                    .replace("{percent}", &percent.get().to_string())
                                    .to_string();
                            }
                            ClassData::Color(color) => {
                                class_body = class_body
                                    .replace("{color-lookup}", &color.get().unwrap())
                                    .to_string();
                            }
                        }
                    }

                    return Ok(class_body);
                }
            }
        }

        let mut name_template_suggestions = vec![];

        find_suggestions(
            node_clone,
            class_name_template,
            &mut name_template_suggestions,
        );

        name_template_suggestions.sort();

        return Err(ClassDictionaryError::InvalidClassName {
            name: class_full_name.to_string(),
            name_template_suggestions,
        });
    }
}

fn find_suggestions(
    node: Rc<RefCell<ClassSegmentNode>>,
    name_template_start: &str,
    name_template_suggestions: &mut Vec<String>,
) {
    if name_template_suggestions.len() > MAX_SUGGESTIONS {
        return;
    }

    let node = node.borrow();
    for (_segment_name, segment) in node.children.iter() {
        match &segment.borrow().node {
            ClassSegmentOrCssTemplate::Char(char) => {
                let name_template = name_template_start.to_string() + &char.to_string();
                find_suggestions(segment.clone(), &name_template, name_template_suggestions);
            }
            ClassSegmentOrCssTemplate::SpecialSegment(special) => {
                let name_template = name_template_start.to_string() + &special;
                find_suggestions(segment.clone(), &name_template, name_template_suggestions);
            }
            ClassSegmentOrCssTemplate::CssTemplate(_) => {
                name_template_suggestions.push(name_template_start.to_string());
            }
        };
    }
}

#[cfg(test)]
mod tests {
    use super::*;
    use crate::color::{ColorName, CssColor};
    use crate::SunbeamConfig;

    /// Verify that we can parse a class that does not have any dynamic values (such as pixels)
    /// in it.
    #[test]
    fn parse_static_class() {
        let class = ClassDictionary::new()
            .get_class_definition("text-decoration-line-none", &SunbeamConfig::default())
            .unwrap();
        assert_eq!(
            class.css_class_definition,
            r#".text-decoration-line-none {
    text-decoration-line: none;
}"#
        );
    }

    /// Verify that we can parse a class that has a dynamic number of pixels.
    #[test]
    fn parse_class_with_pixels() {
        let class = ClassDictionary::new()
            .get_class_definition("mb75", &SunbeamConfig::default())
            .unwrap();
        assert_eq!(
            class.css_class_definition,
            r#".mb75 {
    margin-bottom: 75px;
}"#
        );
    }

    /// Verify that we can parse a class that has a dynamic percentage.
    #[test]
    fn parse_class_with_percent() {
        let class = ClassDictionary::new()
            .get_class_definition("w85pc", &SunbeamConfig::default())
            .unwrap();
        assert_eq!(
            class.css_class_definition,
            r#".w85pc {
    width: 85%;
}"#
        );
    }

    /// Verify that we can parse a class that contains the name of a color to lookup.
    #[test]
    fn parse_class_with_color_lookup() {
        let mut config = SunbeamConfig::default();
        config.colors.insert(
            ColorName::new("red-10".to_string()).unwrap(),
            CssColor::new("#ff1122".to_string()).unwrap(),
        );

        let class = ClassDictionary::new()
            .get_class_definition("text-color-red-10", &config)
            .unwrap();
        assert_eq!(
            class.css_class_definition,
            r#".text-color-red-10 {
    color: #ff1122;
}"#
        );
    }

    /// Verify that we can parse the hover: modifier.
    #[test]
    fn parse_hover_modifier() {
        let class = ClassDictionary::new()
            .get_class_definition("hover:mb30", &SunbeamConfig::default())
            .unwrap();
        assert_eq!(
            class.css_class_definition,
            r#".hover\:mb30:hover {
    margin-bottom: 30px;
}"#
        );
    }

    /// Verify that we can parse a class that contains a gteq screen width modifier.
    #[test]
    fn parse_gteq_screen_width_modifier() {
        let mut config = SunbeamConfig::default();
        config.screen_widths.insert(640);

        let class = ClassDictionary::new()
            .get_class_definition("gteq640:mb30", &config)
            .unwrap();
        assert_eq!(
            class.css_class_definition,
            r#"
@media (min-width: 640px) {
    .gteq640\:mb30 {
        margin-bottom: 30px;
    }
}"#
            .trim()
        );
    }

    /// Verify that we return an error if the class name does not match any of the class name
    /// templates.
    #[test]
    fn error_if_class_name_invalid() {
        match ClassDictionary::new()
            .get_class_definition("wqq30", &SunbeamConfig::default())
            .err()
            .unwrap()
        {
            ClassDictionaryError::InvalidClassName {
                name,
                name_template_suggestions,
            } => {
                assert_eq!(name, "wqq30");

                assert!(name_template_suggestions.len() >= 2);
                let mut has_width_pixels_suggestion = false;
                let mut has_width_percent_suggestion = false;

                for suggestion in name_template_suggestions {
                    if suggestion == "w{pixels}" {
                        has_width_pixels_suggestion = true;
                    } else if suggestion == "w{percent}pc" {
                        has_width_percent_suggestion = true;
                    }
                }

                assert!(has_width_pixels_suggestion);
                assert!(has_width_percent_suggestion);
            }
        };
    }
}
