use super::{ClassDefinition, ClassDictionary};
use crate::class_dictionary::{ClassSegmentNode, ClassSegmentOrCssTemplate};
use std::cell::RefCell;
use std::rc::Rc;

impl ClassDictionary {
    pub(crate) fn build_trie(
        parent: &Rc<RefCell<ClassSegmentNode>>,
        class_slice: &str,
        class_def: &ClassDefinition,
    ) {
        if class_slice.len() == 0 {
            Self::insert_css_template_node(parent, class_def);
            return;
        }

        if class_slice.starts_with("{") {
            let closing_bracket_idx = class_slice.find("}").unwrap();
            let bracketed = &class_slice[0..=closing_bracket_idx];

            let pixels_node = ClassSegmentNode {
                node: ClassSegmentOrCssTemplate::SpecialSegment(bracketed.to_string()),
                children: Default::default(),
            };
            let mut parent = parent.borrow_mut();
            let pixels_node = parent
                .children
                .entry(bracketed.to_string())
                .or_insert(Rc::new(RefCell::new(pixels_node)));

            Self::build_trie(
                pixels_node,
                &class_slice[closing_bracket_idx + 1..],
                class_def,
            );
            return;
        }

        let char = &class_slice[0..1];
        let char = char.chars().next().unwrap();
        let char_node = ClassSegmentNode {
            node: ClassSegmentOrCssTemplate::Char(char),
            children: Default::default(),
        };
        let mut parent = parent.borrow_mut();
        let child = parent
            .children
            .entry(char.to_string())
            .or_insert(Rc::new(RefCell::new(char_node)));

        Self::build_trie(child, &class_slice[1..], class_def);
    }

    fn insert_css_template_node(
        parent: &Rc<RefCell<ClassSegmentNode>>,
        class_def: &ClassDefinition,
    ) {
        let mut parent = parent.borrow_mut();
        parent.children.insert(
            "".to_string(),
            Rc::new(RefCell::new(ClassSegmentNode {
                node: ClassSegmentOrCssTemplate::CssTemplate(class_def.css.clone()),
                children: Default::default(),
            })),
        );
    }
}
