use std::cell::RefCell;
use std::collections::HashMap;
use std::rc::Rc;

mod build_trie;
mod get_class_definition;

// TODO: Make use of this in our parsing code (so, make this pub(crate))
#[derive(Debug)]
pub(crate) struct ClassDictionary {
    root: Rc<RefCell<ClassSegmentNode>>,
}

impl ClassDictionary {
    pub(crate) fn new() -> Self {
        let class_definitions: ClassDefinitions = serde_yaml::from_str(CLASSES_YAML).unwrap();

        let root = ClassSegmentNode {
            node: ClassSegmentOrCssTemplate::SpecialSegment("".to_string()),
            children: HashMap::new(),
        };
        let root = Rc::new(RefCell::new(root));

        for (class_name, class_def) in class_definitions.class_definitions {
            Self::build_trie(&root, &class_name, &class_def);
        }

        let dictionary = ClassDictionary { root };
        dictionary
    }
}

#[derive(serde::Deserialize)]
struct ClassDefinitions {
    class_definitions: HashMap<String, ClassDefinition>,
}
#[derive(serde::Deserialize)]
pub(crate) struct ClassDefinition {
    css: String,
}

const CLASSES_YAML: &'static str = r#"
class_definitions:
  text-decoration-line-none:
    css: "text-decoration-line: none;"
  mb{pixels}:
    css: "margin-bottom: {pixels}px;"
  w{percent}pc:
    css: "width: {percent}%;"
  w{pixels}:
    css: "width: {pixels}px;"
  max-w-{pixels}:
    css: "max-width: {pixels}px;"
  h{percent}pc:
    css: "height: {percent}%;"
  h{pixels}:
    css: "height: {pixels}px;"
  text-color-{color}:
    css: "color: {color-lookup};"
  p{pixels}:
    css: "padding: {pixels}px;"
  pt{pixels}:
    css: "padding-top: {pixels}px;"
  pl{pixels}:
    css: "padding-left: {pixels}px;"
  pb{pixels}:
    css: "padding-bottom: {pixels}px;"
  pr{pixels}:
    css: "padding-right: {pixels}px;"
  mt{pixels}:
    css: "margin-top: {pixels}px;"
  mb{pixels}:
    css: "margin-bottom: {pixels}px;"
  ml{pixels}:
    css: "margin-left: {pixels}px;"
  mr{pixels}:
    css: "margin-right: {pixels}px;"
  overflow-y-scroll:
    css: "overflow-y: scroll;"
  overflow-y-auto:
    css: "overflow-y: auto;"
  cursor-pointer:
    css: "cursor: pointer;"
  flex-dir-col:
    css: "flex-direction: column;"
  display-flex:
    css: "display: flex;"
  box-sizing-border-box:
    css: "box-sizing: border-box;"
  border-width-{pixels}:
    css: "border-width: {pixels}px;"
  border-right-width-{pixels}:
    css: "border-right-width: {pixels}px;"
  border-bottom-width-{pixels}:
    css: "border-bottom-width: {pixels}px;"
  border-left-width-{pixels}:
    css: "border-left-width: {pixels}px;"
  border-top-width-{pixels}:
    css: "border-top-width: {pixels}px;"
  border-style-solid:
    css: "border-style: solid;"
  bg-color-{color}:
    css: "background-color: {color-lookup};"
  flex-1:
    css: "flex: 1;"
  align-items-center:
    css: "align-items: center;"
  border-color-{color}:
    css: "border-color: {color-lookup};"
"#;

#[derive(Debug)]
enum ClassSegmentOrCssTemplate {
    Char(char),
    // TODO: One variant per special segment kind...
    SpecialSegment(String),
    CssTemplate(String),
}

#[derive(Debug)]
pub(crate) struct ClassSegmentNode {
    node: ClassSegmentOrCssTemplate,
    children: HashMap<String, Rc<RefCell<ClassSegmentNode>>>,
}

/// An error when interfacing with the class dictionary.
#[derive(Debug, thiserror::Error)]
pub enum ClassDictionaryError {
    #[allow(missing_docs)]
    #[error("Class name {name} is invalid.\nDid you mean: {name_template_suggestions:?}")]
    InvalidClassName {
        /// The name that was provided.
        name: String,
        /// A few suggestions for valid class names.
        name_template_suggestions: Vec<String>,
    },
}
