/// Modifiers for the class, such as "hover:" or "visited:"
#[derive(Debug, Eq, PartialEq, Ord, PartialOrd, Hash, Copy, Clone, Default)]
pub struct Modifiers {
    /// ex: hover:mr10
    hover: bool,
    /// ex: visited:mr10
    visited: bool,
    /// Put this class behind a min-width media query.
    /// ex: gteq640:mt5
    min_width_gteq: Option<u32>,
}

/// An error while setting modifiers.
#[derive(Debug, thiserror::Error)]
pub enum ModifierError {
    /// Modifier was already enabled.
    #[error("The {0:?} modifier was already set.")]
    AlreadySet(ModifierKind),
}

/// The type of modifier.
#[derive(Debug, Copy, Clone)]
pub enum ModifierKind {
    /// hover:
    Hover,
    /// visited:
    Visited,
    /// gteq:
    ScreenWidthGteq,
}

impl Modifiers {
    /// Set that the hover modifier is enabled.
    pub fn set_hover_enabled(&mut self) -> Result<(), ModifierError> {
        if self.hover {
            return Err(ModifierError::AlreadySet(ModifierKind::Hover));
        }

        self.hover = true;
        Ok(())
    }

    /// Set that the visited modifier is enabled.
    pub fn set_visited_enabled(&mut self) -> Result<(), ModifierError> {
        if self.hover {
            return Err(ModifierError::AlreadySet(ModifierKind::Visited));
        }

        self.visited = true;
        Ok(())
    }

    /// Set the media query screen width >=
    pub fn set_min_width_gteq(&mut self, screen_width: u32) -> Result<(), ModifierError> {
        if self.hover {
            return Err(ModifierError::AlreadySet(ModifierKind::Visited));
        }

        self.min_width_gteq = Some(screen_width);
        Ok(())
    }

    pub(crate) fn add_modifiers_to_class_name(&mut self, class_name: &str) -> String {
        let maybe_hover = if self.hover { ":hover" } else { "" };
        let maybe_visited = if self.visited { ":visited" } else { "" };

        format!(
            "{class_name}{maybe_hover}{maybe_visited}",
            class_name = class_name,
            maybe_hover = maybe_hover,
            maybe_visited = maybe_visited
        )
    }
}

impl Modifiers {
    /// Whether or not the hover modifier is enabled.
    pub fn hover(&self) -> bool {
        self.hover
    }

    /// Whether or not the visited modifier is enabled.
    pub fn visited(&self) -> bool {
        self.visited
    }

    /// Whether or not the media query min-width modifier is enabled.
    pub fn min_width_gteq(&self) -> Option<u32> {
        self.min_width_gteq
    }
}
