# Sudograph settings

There will be many settings that Sudograph will allow the developer to customize. Sudograph settings are set in your GraphQL schema using the `SudographSettings` object type. The following are supported now:

```graphql
type SudographSettings {
    exportGeneratedQueryFunction: true
    exportGeneratedMutationFunction: true
    exportGeneratedInitFunction: true
    exportGeneratedPostUpgradeFunction: true
}
```

# exportGeneratedQueryFunction

Defaults to `true`. If set to `false`, the `graphql_query` function generated by Sudograph will not be exported as a publicly available canister function. This would allow you to implement your own logic before executing a query, for example as part of an authorization flow.

Here's an example of overriding the generated `graphql_query` function with some basic authorization. You would create the following GraphQL schema in `canisters/graphql/src/schema.graphql`:

```graphql
type SudographSettings {
    exportGeneratedQueryFunction: false
}

type User {
    id: ID!
}
```

You would write the following in `canisters/graphql/src/graphql.rs`:

```rust
use sudograph::graphql_database;

graphql_database!("canisters/graphql/src/schema.graphql");

#[sudograph::ic_cdk_macros::query]
async fn graphql_query_custom(query_string: String, variables_json_string: String) -> String {
    let authorized_principal = sudograph::ic_cdk::export::Principal::from_text("y6lgw-chi3g-2ok7i-75s5h-k34kj-ybcke-oq4nb-u4i7z-vclk4-hcpxa-hqe").expect("should be able to decode");
    
    if sudograph::ic_cdk::caller() != authorized_principal {
        panic!("Not authorized");
    }

    return graphql_query(query_string, variables_json_string).await;
}
```

You would update `canisters/graphql/src/graphql.did`:

```
service : {
    "graphql_query_custom": (text, text) -> (text) query;
    "graphql_mutation": (text, text) -> (text);
}
```

# exportGeneratedMutationFunction

Defaults to `true`. If set to `false`, the `graphql_mutation` function generated by Sudograph will not be exported as a publicly available canister function. This would allow you to implement your own logic before executing a mutation, for example as part of an authorization flow.

Here's an example of overriding the generated `graphql_mutation` function with some basic authorization. You would create the following GraphQL schema in `canisters/graphql/src/schema.graphql`:

```graphql
type SudographSettings {
    exportGeneratedMutationFunction: false
}

type User {
    id: ID!
}
```

You would write the following in `canisters/graphql/src/graphql.rs`:

```rust
use sudograph::graphql_database;

graphql_database!("canisters/graphql/src/schema.graphql");

#[sudograph::ic_cdk_macros::update]
async fn graphql_mutation_custom(mutation_string: String, variables_json_string: String) -> String {
    let authorized_principal = sudograph::ic_cdk::export::Principal::from_text("y6lgw-chi3g-2ok7i-75s5h-k34kj-ybcke-oq4nb-u4i7z-vclk4-hcpxa-hqe").expect("should be able to decode");
    
    if sudograph::ic_cdk::caller() != authorized_principal {
        panic!("Not authorized");
    }

    return graphql_mutation(mutation_string, variables_json_string).await;
}
```

You would update `canisters/graphql/src/graphql.did`:

```
service : {
    "graphql_query": (text, text) -> (text) query;
    "graphql_mutation_custom": (text, text) -> (text);
}
```

# exportGeneratedInitFunction

Defaults to `true`. If set to `false`, the `init` function generated by Sudograph will not be exported as a publicly available canister function. This would allow you to implement your own logic during canister initialization. You'll want to make sure to call the generated `init` function after your functionality is complete, as it executes all of the `init` mutations that initialize the database.

Here's an example of overriding the generated `init` function. You would create the following GraphQL schema in `canisters/graphql/src/schema.graphql`:

```graphql
type SudographSettings {
    exportGeneratedInitFunction: false
}

type User {
    id: ID!
}
```

You would write the following in `canisters/graphql/src/graphql.rs`:

```rust
use sudograph::graphql_database;

graphql_database!("canisters/graphql/src/schema.graphql");

#[sudograph::ic_cdk_macros::init]
async fn init_custom() {
    init.await;
}
```

# exportGeneratedPostUpgradeFunction

Defaults to `true`. If set to `false`, the `post_upgrade` function generated by Sudograph will not be exported as a publicly available canister function. This would allow you to implement your own logic during canister post upgrade. You'll want to make sure to call the generated `post_upgrade` function after your functionality is complete, as it executes all of the `init` mutations that initialize the database (unless you are keeping your state through stable memory, then you would not want to initialize the database again).

Here's an example of overriding the generated `post_upgrade` function. You would create the following GraphQL schema in `canisters/graphql/src/schema.graphql`:

```graphql
type SudographSettings {
    exportGeneratedPostUpgradeFunction: false
}

type User {
    id: ID!
}
```

You would write the following in `canisters/graphql/src/graphql.rs`:

```rust
use sudograph::graphql_database;

graphql_database!("canisters/graphql/src/schema.graphql");

#[sudograph::ic_cdk_macros::post_upgrade]
async fn post_upgrade_custom() {
    post_upgrade.await;
}
```