/*
   Copyright 2019 Supercomputing Systems AG

   Licensed under the Apache License, Version 2.0 (the "License");
   you may not use this file except in compliance with the License.
   You may obtain a copy of the License at

       http://www.apache.org/licenses/LICENSE-2.0

   Unless required by applicable law or agreed to in writing, software
   distributed under the License is distributed on an "AS IS" BASIS,
   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
   See the License for the specific language governing permissions and
   limitations under the License.

*/

//! Offers macros that build extrinsics for custom runtime modules based on the metadata.
//! Additionally, some predefined extrinsics for common runtime modules are implemented.

pub mod xt_primitives;

use crate::{Api, Index, Metadata};
use codec::Encode;
use sp_core::crypto::Pair;
use sp_core::H256;
use sp_runtime::generic::Era;
use sp_runtime::MultiSignature;
use xt_primitives::*;

type Call<A> = (u8, u8, A);

/// Generates the extrinsic's call field for a given module and call passed as &str
pub fn compose_call<A: Encode>(metadata: &Metadata, module: &str, call: &str, args: A) -> Call<A> {
    let module = metadata.module_with_calls(module).unwrap();
    let call_index = module.calls.get(call).unwrap();
    (module.index, *call_index as u8, args)
}

/// Generates an Unchecked extrinsic for a given call
pub fn compose_extrinsic_offline<P, A>(
    signer: &P,
    call: Call<A>,
    nonce: Index,
    era: Era,
    genesis_hash: H256,
    genesis_or_current_hash: H256,
    runtime_spec_version: u32,
    transaction_version: u32,
) -> UncheckedExtrinsicV4<Call<A>>
where
    P: Pair,
    MultiSignature: From<P::Signature>,
    A: Encode,
{
    let extra = GenericExtra::new(era, nonce);
    let raw_payload = SignedPayload::from_raw(
        &call,
        &extra,
        (
            runtime_spec_version,
            transaction_version,
            genesis_hash,
            genesis_or_current_hash,
            (),
            (),
            (),
        ),
    );

    let signature = raw_payload.using_encoded(|payload| signer.sign(payload));

    let mut arr: [u8; 32] = Default::default();
    arr.clone_from_slice(signer.public().as_ref());

    UncheckedExtrinsicV4::new_signed(
        call,
        GenericAddress::from(AccountId::from(arr)),
        signature.into(),
        extra,
    )
}

/// Generates an Unchecked extrinsic for a given module and call passed as a &str.
pub fn compose_extrinsic<P, A>(
    api: &Api<P>,
    module: &str,
    call: &str,
    args: A,
) -> UncheckedExtrinsicV4<Call<A>>
where
    P: Pair,
    MultiSignature: From<P::Signature>,
    A: Encode,
{
    let call = compose_call(&api.metadata, module, call, args);

    if let Some(signer) = &api.signer {
        compose_extrinsic_offline(
            signer,
            call,
            api.get_nonce().unwrap(),
            Era::Immortal,
            api.genesis_hash,
            api.genesis_hash,
            api.runtime_version.spec_version,
            api.runtime_version.transaction_version,
        )
    } else {
        UncheckedExtrinsicV4 {
            signature: None,
            function: call,
        }
    }
}
