#![no_std]

//////////////////////////////////////////////

extern crate alloc;

use alloc::format;
use alloc::string::ToString;
use proc_macro::TokenStream;
use proc_macro2::Span;
use quote::quote;
use syn::{parse_macro_input, Ident, ItemFn, AttributeArgs, Lit, NestedMeta};

//////////////////////////////////////////////

/// Define a unit test that should run with Substance.
///
/// The function with this attribute will be marked as a Substance unit test function and the current
/// macro notation will be replaced by the appropriate object for Substance and marked for cargo to see
/// it as a test to be provided to the runner.
///
/// Please note that the testing function should respect that signature: `fn _() {}` without any
/// parameters nor return value (like with Cargo default test framework).
///
/// The macro support two arguments (for now) that enable you to specify if a test should be ignored
/// or if a test is expected to fail. At the moment, they are both simple String that must respect
/// the precise writing (WIP).
///
/// # Examples
///
/// To define a test function:
///
/// ```rust
/// #[substance_test]
/// fn my_test() {
///     [...]
/// }
/// ```
///
/// To define a test function that should panic:
///
/// ```rust
/// #[substance_test("Should_panic")]
/// fn my_test_that_must_panic() {
///     [...]
/// }
/// ```
///
/// To define a test function that need to be ignored:
///
/// ```rust
/// #[substance_test("Ignore")]
/// fn my_ignored_test() {
///     [...]
/// }
/// ```
///
/// You can combine both too (order doesn't matter):
///
/// ```rust
/// #[substance_test("Ignore", "Should_panic")]
/// fn my_ignored_panicking_test() {
///     [...]
/// }
/// ```
#[proc_macro_attribute]
pub fn substance_test(args: TokenStream, input: TokenStream) -> TokenStream {
    let attr_args = parse_macro_input!(args as AttributeArgs);
    let item_fn = parse_macro_input!(input as ItemFn);
    let ident = item_fn.sig.ident.to_string();

    let test_name = ident.as_str();
    let test_func = item_fn.block;
    let test_ident = Ident::new(format!("__{}_SUBSTANCE_UNIT_TEST", ident.to_uppercase()).as_str(),
                                Span::call_site());

    let mut ignore = false;
    let mut should_panic = false;

    attr_args.iter().for_each(|arg| {
        match arg {
            NestedMeta::Meta(_) => {
                unreachable!("Unsupported attribute of type Meta for `#[substance_test]`");
            }
            NestedMeta::Lit(l) => {
                match l {
                    Lit::Str(s) => {
                        if "Ignore".eq(s.value().as_str()) {
                            ignore = true;
                        }
                        if "Should_panic".eq(s.value().as_str()) {
                            should_panic = true;
                        }
                    }
                    _ => {
                        unreachable!("Unsupported attribute of type Lit for `#[substance_test]`");
                    }
                }
            }
        }
    });

    quote!(
        #[test_case]
        const #test_ident: substance_framework::UnitTest = substance_framework::UnitTest {
            name: #test_name,
            test_func: || #test_func,
            ignored: #ignore,
            should_panic: #should_panic
        };
    ).into()
}