/***********************************************************************************
 * MIT License                                                                     *
 *                                                                                 *
 * Copyright (c) 2022 Tutul                                                        *
 *                                                                                 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy    *
 * of this software and associated documentation files (the "Software"), to deal   *
 * in the Software without restriction, including without limitation the rights    *
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell       *
 * copies of the Software, and to permit persons to whom the Software is           *
 * furnished to do so, subject to the following conditions:                        *
 *                                                                                 *
 * The above copyright notice and this permission notice shall be included in all  *
 * copies or substantial portions of the Software.                                 *
 *                                                                                 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR      *
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,        *
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE     *
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER          *
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,   *
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE   *
 * SOFTWARE.                                                                       *
 ***********************************************************************************/

use crate::{crash, printer::Printer, Outcome, UnitTest, STATE};

//////////////////////////////////////////////

pub(crate) fn run_tests(tests: &'static [&'static UnitTest<'static>]) {
    let state;

    unsafe {
        state = &mut STATE;
    }

    let mut conclusion = state.conclusion.insert(Default::default());

    // Create printer which is used for all output.
    let printer = state.printer.insert(Printer::new(tests));

    // Print number of tests
    printer.print_title(tests.len() as u64);

    if tests.is_empty() {
        printer.no_test();
        return;
    }

    state.tests = Some(tests);
    conclusion.num_skipped = tests.len();

    // Execute all tests
    executor();

    printer.print_summary(conclusion);
}

fn executor() {
    let state;

    unsafe {
        state = &mut STATE;
    }

    if state.conclusion.is_none() || state.printer.is_none() || state.tests.is_none() {
        crash(
            file!(),
            line!(),
            column!(),
            Some(&format_args!(
                "Environment isn't properly setup (C:{}, P:{}, T:{})",
                state.conclusion.is_some(),
                state.printer.is_some(),
                state.tests.is_some()
            )),
        );
    }

    let conclusion = state.conclusion.as_mut().unwrap();
    let printer = state.printer.as_mut().unwrap();

    state.tests.unwrap().iter().for_each(|t| {
        let test = state.current.insert(t);
        let outcome;

        printer.print_test(test.name);

        if test.ignored {
            outcome = Outcome::Ignored;
            conclusion.num_ignored += 1;
        } else {
            state.panic = false;

            // Run the given function
            (test.test_func)();

            if state.panic == test.should_panic {
                outcome = Outcome::Passed;
                conclusion.num_passed += 1;
            } else {
                outcome = Outcome::Failed { reason: state.msg };
                conclusion.num_failed += 1;
            }
        };

        printer.print_single_outcome(&outcome);
        conclusion.num_skipped -= 1;
        state.counter += 1;

        let _drop = state.current.take();
    });
}
