/***********************************************************************************
 * MIT License                                                                     *
 *                                                                                 *
 * Copyright (c) 2022 Tutul                                                        *
 *                                                                                 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy    *
 * of this software and associated documentation files (the "Software"), to deal   *
 * in the Software without restriction, including without limitation the rights    *
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell       *
 * copies of the Software, and to permit persons to whom the Software is           *
 * furnished to do so, subject to the following conditions:                        *
 *                                                                                 *
 * The above copyright notice and this permission notice shall be included in all  *
 * copies or substantial portions of the Software.                                 *
 *                                                                                 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR      *
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,        *
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE     *
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER          *
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,   *
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE   *
 * SOFTWARE.                                                                       *
 ***********************************************************************************/

#![feature(lang_items)]
#![feature(core_intrinsics)]
#![feature(panic_internals)]
#![feature(panic_info_message)]
#![feature(stmt_expr_attributes)]
#![no_std]

//////////////////////////////////////////////

use core::fmt::Arguments;
pub use core::intrinsics::abort;
pub use core::panic::{Location, PanicInfo};

//////////////////////////////////////////////

#[macro_use]
extern crate libc_print;

#[doc(hidden)]
pub mod macros;
#[cfg(not(doc))]
pub mod panic;
mod printer;
mod runner;

//////////////////////////////////////////////

#[cfg(not(doc))]
pub fn crash(file: &str, line: u32, column: u32, arguments: Option<&Arguments>) -> ! {
    unsafe {
        panic::cx_panic(&PanicInfo::internal_constructor(
            arguments,
            &Location::internal_constructor(file, line, column),
            false,
        ))
    }
}

#[cfg(doc)]
#[doc(hidden)]
pub fn crash(file: &str, line: u32, column: u32, arguments: Option<&Arguments>) -> ! {
    loop {}
}

//////////////////////////////////////////////

pub struct State<'s> {
    pub panic: bool,
    pub msg: Option<&'s str>,
    tests: Option<&'s [&'s UnitTest<'s>]>,
    current: Option<&'s UnitTest<'s>>,
    printer: Option<printer::Printer>,
    conclusion: Option<Conclusion>,
    counter: usize,
}

#[no_mangle]
#[doc(hidden)]
pub static mut STATE: State = State {
    panic: false,
    msg: None,
    tests: None,
    current: None,
    printer: None,
    conclusion: None,
    counter: 0,
};

//////////////////////////////////////////////

/// Represent the outcome of a particular test
///
/// # Ignored
///
/// The test was ignored and didn't run
///
/// # Passed
///
/// The test passed without any problem
///
/// # Failed
///
/// The test didn't pass or an error occurred during it's run
/// If `msg` is provided, a custom message will be added to be shown on the screen
/// You may find more details in `reason` if present
///
/// # Aborted
///
/// The test was aborted for some reason
/// It is usually a sign that the test isn't parsable by Substance
/// You may find more details in `reason` if present
///
/// # Skipped
///
/// The test wasn't able to run because the previous one was aborted
#[derive(PartialEq, Eq)]
pub enum Outcome<'o> {
    Ignored,
    Passed,
    Failed { reason: Option<&'o str> },
    Aborted { reason: Option<&'o str> },
    Skipped,
}

//////////////////////////////////////////////

/// Struct that contain information about a test function to run
pub struct UnitTest<'u> {
    pub name: &'u str,
    pub test_func: fn(),
    pub ignored: bool,
    pub should_panic: bool,
}

//////////////////////////////////////////////

#[derive(Default)]
pub struct Conclusion {
    has_aborted: bool,
    num_ignored: usize,
    num_passed: usize,
    num_failed: usize,
    num_skipped: usize,
}

//////////////////////////////////////////////

pub fn test_runner(tests: &'static [&'static UnitTest<'static>]) {
    crate::runner::run_tests(tests);
}
