//////////////////////////////////////////////

use crate::{crash, Outcome, printer::Printer, STATE, UnitTest};

//////////////////////////////////////////////

pub(crate) fn run_tests(tests: &'static [&'static UnitTest<'static>]) {
    let state;

    unsafe { state = &mut STATE; }

    let mut conclusion = state.conclusion.insert(Default::default());

    // Create printer which is used for all output.
    let printer = state.printer.insert(Printer::new(tests));

    // Print number of tests
    printer.print_title(tests.len() as u64);

    if tests.is_empty() {
        printer.no_test();
        return;
    }

    state.tests = Some(tests);
    conclusion.num_skipped = tests.len();

    // Execute all tests
    executor();

    printer.print_summary(conclusion);
}

fn executor() {
    let state;

    unsafe {
        state = &mut STATE;
    }

    if state.conclusion.is_none() || state.printer.is_none() || state.tests.is_none() {
        crash(file!(), line!(), column!(),
              Some(&format_args!("Environment isn't properly setup (C:{}, P:{}, T:{})",
                                 state.conclusion.is_some(),
                                 state.printer.is_some(),
                                 state.tests.is_some())));
    }

    let conclusion = state.conclusion.as_mut().unwrap();
    let printer = state.printer.as_mut().unwrap();

    state.tests.unwrap().iter().for_each(|t| {
        let test = state.current.insert(t);
        let outcome;

        printer.print_test(test.name);

        if test.ignored {
            outcome = Outcome::Ignored;
            conclusion.num_ignored += 1;
        } else {
            state.panic = false;

            // Run the given function
            (test.test_func)();

            if state.panic == test.should_panic {
                outcome = Outcome::Passed;
                conclusion.num_passed += 1;
            } else {
                outcome = Outcome::Failed {reason: state.msg };
                conclusion.num_failed += 1;
            }
        };

        printer.print_single_outcome(&outcome);
        conclusion.num_skipped -= 1;
        state.counter += 1;

        let _drop = state.current.take();
    });
}
