pub use core::panic::{Location, PanicInfo};
pub use core::intrinsics::abort;
use core::intrinsics::likely;

//////////////////////////////////////////////

use crate::{Conclusion, Outcome, STATE};

//////////////////////////////////////////////

#[panic_handler]
#[no_mangle]
pub unsafe fn cx_panic(info: &PanicInfo) -> ! {
    let location = info.location();
    let message = info.message();

    if likely(STATE.printer.is_some()) {
        let printer = STATE.printer.as_mut().unwrap();
        let mut default = Conclusion { has_aborted: true, num_ignored: usize::MAX, num_passed: 0, num_failed: usize::MAX, num_skipped: usize::MAX };
        let mut conclusion = match STATE.conclusion.as_mut() {
            Some(c) => {
                c.has_aborted = true;
                c
            },
            None => &mut default};

        if likely(STATE.current.is_some()) {
            let outcome = Outcome::Aborted { reason: match message {
                Some(msg) => msg.as_str(),
                None => None
            } };
            conclusion.num_skipped -= 1;
            STATE.counter += 1;
            printer.print_single_outcome(&outcome);
        }

        let outcome = Outcome::Skipped;
        for index in STATE.counter .. STATE.tests.unwrap().len() {
            printer.print_test(STATE.tests.unwrap()[index].name);
            printer.print_single_outcome(&outcome);
        }

        printer.print_summary(conclusion);
    }

    libc_println!();

    if likely(message.is_some() && location.is_some()) {
        libc_println!("[PANIC] {}:{}:{}", location.unwrap().file(), location.unwrap().line(), location.unwrap().column());
        libc_println!("{}", message.unwrap());
    } else if message.is_some() {
        libc_println!("[PANIC] {} (unknown location)", message.unwrap());
    } else if location.is_some() {
        libc_println!("[PANIC] A panic occurred at {}:{}:{}", location.unwrap().file(),
            location.unwrap().line(), location.unwrap().column());
    } else {
        libc_println!("[PANIC] A panic occurred but not information available");
        libc_println!("    >>> {:?}", info);
    }
    libc_println!();

    abort()
}

#[lang = "eh_personality"]
#[no_mangle]
pub unsafe fn rust_eh_personality() -> ! {
    unreachable!("eh_personality")
}

#[allow(non_snake_case)]
#[no_mangle]
pub unsafe fn _Unwind_Resume() -> ! {
    unreachable!("_Unwind_Resume")
}