/// Provide a panic-like macro that can be used in the framework to detect a test failure and save
/// that status with a message for later screen printing.
///
/// std unit tests rely heavily on unwinding. Each unit test is run inside a catch_unwind block and
/// if the unit test panics then the panic is caught and the test is marked as failed
/// (or as passed if the unit test was marked with `#[should_panic]`).
///
/// The framework attempt to emulate this behavior by by-passing the panic macro to mark the test
/// failed and then early return instead of unwind. Of course, this emulation doesn't work on panic!
/// originates from outside the crate under test, because panic is not overridden in that scope.
/// This macro provide a way to differentiate between true panic and test failure one.
///
/// But this emulation has some limitation. For instance, it can only cause one panicking function to
/// return. To remediate on that, any subsequent panic! call will directly return causing some panic
/// to be invisible until they get the chance to be the first one. To provide a bit of insight, the
/// macro also set a `panicking` flag to let you know that an other panic was thrown after the first
/// one (but without any messages or context). If a panicking panic is triggered again, the framework
/// will considered either the tests ill written or the failure too fatal and cause a real panic to
/// cause an abort!
#[macro_export]
macro_rules! substance_panic {
    () => (
        substance_panic!("explicit panic")
    );
    ($fmt:expr) => ({
        substance_panic!($fmt,)
    });
    ($fmt:expr, $($arg:tt)*) => ({
        #[allow(improper_ctypes)]
        extern "Rust" {
            static mut STATE: substance_framework::State<'static>;
        }
        unsafe {
            if STATE.panic {
                substance_framework::crash(file!(), line!(), column!(), Some(&format_args!("{}", "panic is panicking again")));
            } else {
                STATE.panic = true;
                STATE.msg = format_args!($fmt, $($arg)*).as_str();
                return
            }
        }
    });
}

//////////////////////////////////////////////

/// Bypass the default panic macro and redirect all its call to our custom implementation. This is
/// necessary to prevent unwinding and let us detect test failure with some assertions.
///
/// cfr. [substance_panic!]
#[macro_export]
macro_rules! panic {
    ($($tt:tt)*) => {
        substance_panic!($($tt)*);
    };
}

//////////////////////////////////////////////

// #[macro_export]
// macro_rules! init_substance {
//     () => {
//         compile_error!("Missing size");
//     };
//     ($cond:expr) => ({
//         match (&$cond) {
//             size => {
//                 let mut outcomes_var : &'static mut [&'static mut Option<Outcome<'static>>] = [None; *size];
//
//                 #[no_mangle]
//                 pub static mut OUTCOMES = &mut outcomes_var;
//             }
//         }
//     });
// }

//////////////////////////////////////////////

/// Passes if Condition is true.
///
/// Evaluates the condition and passes if it is true. Otherwise the test is marked as failure.
/// The optional string is printed on failure.
///
/// # Examples
///
/// ```rust
/// let my_variable = 37;
/// cx_assert(my_variable == 37);
/// ```
///
/// ```rust
/// fn my_function() -> bool {
///     return false;
/// }
///
/// cx_assert(my_function(), "this test is a failure");
/// ```
#[macro_export]
macro_rules! cx_assert {
    ($cond:expr $(,)?) => ({
        match (&$cond) {
            cond_bool => {
                if ! *cond_bool {
                    substance_panic!("assertion failed: {} is true", &*cond_bool);
                }
            }
        }
    });
    ($cond:expr, $($arg:tt)+) => ({
        match (&$cond) {
            cond_bool => {
                if ! *cond_bool {
                    substance_panic!("assertion failed: {} is true  >> {}", &*cond_bool, format_args!($($arg)+));
                }
            }
        }
    });
}

//////////////////////////////////////////////

/// Passes if both argument are equals.
///
/// Passes if the left argument is equal to the right argument. Otherwise the test is marked as failure.
///
/// # Examples
///
/// ```rust
/// let left = 37;
/// let right = 37;
/// cx_assert_eq(left, right);
/// ```
///
/// ```rust
/// cx_assert_eq(0, 5, "this test is a failure");
/// ```
#[macro_export]
macro_rules! cx_assert_eq {
    ($left:expr, $right:expr $(,)?) => ({
        match (&$left, &$right) {
            (left_val, right_val) => {
                if !(*left_val == *right_val) {
                    substance_panic!("assertion failed: {} == {}", &*left_val, &*right_val);
                }
            }
        }
    });
    ($left:expr, $right:expr, $($arg:tt)+) => ({
        match (&$left, &$right) {
            (left_val, right_val) => {
                if !(*left_val == *right_val) {
                    substance_panic!("assertion failed: {} == {}  >> {}", &*left_val, &*right_val, format_args!($($arg)+));
                }
            }
        }
    });
}

//////////////////////////////////////////////

/// Passes if both argument are not equals.
///
/// Passes if the left argument is not equal to the right argument. Otherwise the test is marked as failure.
///
/// # Examples
///
/// ```rust
/// let left = 37;
/// let right = 42;
/// cx_assert_ne(left, right);
/// ```
///
/// ```rust
/// cx_assert_ne(5, 5, "this test is a failure");
/// ```
#[macro_export]
macro_rules! cx_assert_ne {
    ($left:expr, $right:expr $(,)?) => ({
        match (&$left, &$right) {
            (left_val, right_val) => {
                if !(*left_val != *right_val) {
                    substance_panic!("assertion failed: {} != {}", &*left_val, &*right_val);
                }
            }
        }
    });
    ($left:expr, $right:expr, $($arg:tt)+) => ({
        match (&($left), &($right)) {
            (left_val, right_val) => {
                if !(*left_val != *right_val) {
                    substance_panic!("assertion failed: {} != {}  >> {}", &*left_val, &*right_val, format_args!($($arg)+));
                }
            }
        }
    });
}

//////////////////////////////////////////////

/// Passes if the left argument is smaller than the right one.
///
/// Passes if the left argument is smaller than the right argument. Otherwise the test is marked as failure.
///
/// # Examples
///
/// ```rust
/// let left = -325;
/// let right = 37;
/// cx_assert_lt(left, right);
/// ```
///
/// ```rust
/// cx_assert_lt(0, -3, "this test is a failure");
/// ```
#[macro_export]
macro_rules! cx_assert_lt {
    ($left:expr, $right:expr $(,)?) => ({
        match (&$left, &$right) {
            (left_val, right_val) => {
                if !(*left_val < *right_val) {
                    substance_panic!("assertion failed: {} < {}", &*left_val, &*right_val);
                }
            }
        }
    });
    ($left:expr, $right:expr, $($arg:tt)+) => ({
        match (&($left), &($right)) {
            (left_val, right_val) => {
                if !(*left_val < *right_val) {
                    substance_panic!("assertion failed: {} < {}  >> {}", &*left_val, &*right_val, format_args!($($arg)+));
                }
            }
        }
    });
}

//////////////////////////////////////////////

/// Passes if the left argument is smaller or equals than the right one.
///
/// Passes if the left argument is smaller or equals than the right argument. Otherwise the test is marked as failure.
///
/// # Examples
///
/// ```rust
/// let left = -325;
/// let right = 37;
/// cx_assert_le(left, right);
/// ```
///
/// ```rust
/// cx_assert_le(0, 0);
/// ```
///
/// ```rust
/// cx_assert_le(0, -3, "this test is a failure");
/// ```
#[macro_export]
macro_rules! cx_assert_le {
    ($left:expr, $right:expr $(,)?) => ({
        match (&$left, &$right) {
            (left_val, right_val) => {
                if !(*left_val <= *right_val) {
                    substance_panic!("assertion failed: {} <= {}", &*left_val, &*right_val);
                }
            }
        }
    });
    ($left:expr, $right:expr, $($arg:tt)+) => ({
        match (&($left), &($right)) {
            (left_val, right_val) => {
                if !(*left_val <= *right_val) {
                    substance_panic!("assertion failed: {} <= {}  >> {}", &*left_val, &*right_val, format_args!($($arg)+));
                }
            }
        }
    });
}

//////////////////////////////////////////////

/// Passes if the left argument is greater than the right one.
///
/// Passes if the left argument is greater than the right argument. Otherwise the test is marked as failure.
///
/// # Examples
///
/// ```rust
/// let left = 325;
/// let right = 37;
/// cx_assert_gt(left, right);
/// ```
///
/// ```rust
/// cx_assert_gt(-6, 32, "this test is a failure");
/// ```
#[macro_export]
macro_rules! cx_assert_gt {
    ($left:expr, $right:expr $(,)?) => ({
        match (&$left, &$right) {
            (left_val, right_val) => {
                if !(*left_val > *right_val) {
                    substance_panic!("assertion failed: {} > {}", &*left_val, &*right_val);
                }
            }
        }
    });
    ($left:expr, $right:expr, $($arg:tt)+) => ({
        match (&($left), &($right)) {
            (left_val, right_val) => {
                if !(*left_val > *right_val) {
                    substance_panic!("assertion failed: {} > {}  >> {}", &*left_val, &*right_val, format_args!($($arg)+));
                }
            }
        }
    });
}

//////////////////////////////////////////////

/// Passes if the left argument is greater or equals than the right one.
///
/// Passes if the left argument is greater or equals than the right argument. Otherwise the test is marked as failure.
///
/// # Examples
///
/// ```rust
/// let left = 325;
/// let right = 37;
/// cx_assert_ge(left, right);
/// ```
///
/// ```rust
/// cx_assert_ge(435, 435);
/// ```
///
/// ```rust
/// cx_assert_ge(45, 645884, "this test is a failure");
/// ```
#[macro_export]
macro_rules! cx_assert_ge {
    ($left:expr, $right:expr $(,)?) => ({
        match (&$left, &$right) {
            (left_val, right_val) => {
                if !(*left_val >= *right_val) {
                    substance_panic!("assertion failed: {} >= {}", &*left_val, &*right_val);
                }
            }
        }
    });
    ($left:expr, $right:expr, $($arg:tt)+) => ({
        match (&($left), &($right)) {
            (left_val, right_val) => {
                if !(*left_val >= *right_val) {
                    substance_panic!("assertion failed: {} >= {}  >> {}", &*left_val, &*right_val, format_args!($($arg)+));
                }
            }
        }
    });
}