#![feature(lang_items)]
#![feature(core_panic)]
#![feature(panic_internals)]
#![feature(core_intrinsics)]
#![feature(in_band_lifetimes)]
#![feature(panic_info_message)]
#![feature(stmt_expr_attributes)]
#![no_std]

//////////////////////////////////////////////

use core::fmt::Arguments;
pub use core::panic::{Location, PanicInfo};
pub use core::intrinsics::abort;

//////////////////////////////////////////////

#[macro_use]
extern crate libc_print;

#[cfg(not(doc))]
pub mod panic;
#[doc(hidden)]
pub mod macros;
mod printer;
mod runner;

//////////////////////////////////////////////

#[cfg(not(doc))]
pub fn crash(file: &str, line: u32, column: u32, arguments: Option<&Arguments>) -> ! {
    unsafe {
        panic::cx_panic(&PanicInfo::internal_constructor(arguments,
                                                  &Location::internal_constructor(file, line, column),
                                                  false))
    }
}

#[cfg(doc)]
#[doc(hidden)]
pub fn crash(file: &str, line: u32, column: u32, arguments: Option<&Arguments>) -> ! {
    loop {};
}

//////////////////////////////////////////////

pub struct State<'s> {
    pub panic: bool,
    pub msg: Option<&'s str>,
    tests: Option<&'s [&'s UnitTest<'s>]>,
    current: Option<&'s UnitTest<'s>>,
    printer: Option<printer::Printer>,
    conclusion: Option<Conclusion>,
    counter: usize
}

#[no_mangle]
#[doc(hidden)]
pub static mut STATE: State = State { panic: false, msg: None, tests: None, current: None, printer: None, conclusion: None, counter: 0 };

//////////////////////////////////////////////

/// Represent the outcome of a particular test
///
/// # Ignored
///
/// The test was ignored and didn't run
///
/// # Passed
///
/// The test passed without any problem
///
/// # Failed
///
/// The test didn't pass or an error occurred during it's run
/// If `msg` is provided, a custom message will be added to be shown on the screen
/// You may find more details in `reason` if present
///
/// # Aborted
///
/// The test was aborted for some reason
/// It is usually a sign that the test isn't parsable by Substance
/// You may find more details in `reason` if present
///
/// # Skipped
///
/// The test wasn't able to run because the previous one was aborted
#[derive(PartialEq, Eq)]
pub enum Outcome<'o> {
    Ignored,
    Passed,
    Failed { reason: Option<&'o str>},
    Aborted { reason: Option<&'o str>},
    Skipped
}

//////////////////////////////////////////////

/// Struct that contain information about a test function to run
pub struct UnitTest<'u> {
    pub name: &'u str,
    pub test_func: fn(),
    pub ignored: bool,
    pub should_panic: bool
}

//////////////////////////////////////////////

#[derive(Default)]
pub struct Conclusion {
    has_aborted: bool,
    num_ignored: usize,
    num_passed: usize,
    num_failed: usize,
    num_skipped: usize,
}

//////////////////////////////////////////////

pub fn test_runner(tests: &'static [&'static UnitTest<'static>]) {
    crate::runner::run_tests(tests);
}