#!/bin/bash

set -xeuo pipefail

usage() {
	echo "Usage: $0 REMOTE VERSION"
	echo ""
	echo "where REMOTE is the remote for gitlab.com and VERSION is the new version number"
	echo ""
}

msg() {
	echo "$*"
}

die() {
	echo "ERROR: $*" 1>&2
	exit 1
}

current_branch() {
	git status --branch --short | awk '/^##/ { print $2 }'
}

crates() {
	git ls-files | grep Cargo.toml | sed 's,Cargo\.toml$,,' | sed 's,^$,.,'
}

# [ "$(current_branch)" == main ] || die "current branch is not 'main'"

[ "$#" = 2 ] || (
	usage
	die " command line arguments"
)
remote="$1"
version="$2"

if ! echo "$version" | grep -E '^[0-9]+\.[0-9]+\.[0-9]+$' >/dev/null; then
	die "version number must be numbers and dots only"
fi

# Check that we can publish.
msg "checking that all crates seem OK"
cargo check -q --release --workspace
cargo build -q --release --workspace

# This is disabled, for now, because it fails, because ordering of
# crate publishing matters. Also, a crate can't be mock-published,
# until it's dependencies are published.
if false; then
	msg "checking that all crates can be published"
	for crate in $(crates); do
		(cd "$crate" && msg "mock publishing in $(pwd)" && cargo publish -q --dry-run)
	done
fi

# The actual release.
msg "making actual release"
git tag -sam "Subplot release $version" "$version"
git push --tags "$remote"
cargo publish

msg ""
msg "Subplot version $version has been tagged and published on "
msg "gitlab.com and crates.io. Ask Lars or Daniel to push the tag"
msg "git.liw.fi. Please tell everyone about the new hotness."
