use num_bigint::{BigInt, BigUint};
use num_complex::Complex64;
use num_irrational::{BigQuadratic, Quadratic64};
use num_modular::{MontgomeryBigint, MontgomeryInt};
use num_rational::{BigRational, Rational64};
use num_traits::{FromPrimitive, ToPrimitive};
use std::convert::TryInto;

/// A general `suan` object (su-bject) containing all possible numeric types involved in the calculation
#[derive(Clone)]
pub enum Subject {
    /// Positive infinity (true) or negative infinity (false).
    /// This number should only be used as a sentinel in comparison,
    /// arithmetic operators will panic if one of the operands is infinite.
    Infinity(bool),
    /// Small integer
    Int(i64),
    /// Multi-precision integer
    BInt(BigInt),
    /// Small real number
    Real(f64),
    /// Multi-precision real number
    BReal(), // TODO: available if enable rug feature
    /// Rational number
    Rational(Rational64),
    /// Multi-precision Rational number
    BRational(BigRational),
    /// Complex number
    Complex(Complex64),
    /// Multi-precision Complex number
    BComplex(), // TODO: available if enable rug feature
    /// Small integer in modulo ring
    IntMod(MontgomeryInt<u64>),
    /// Multi-precision integer in modulo ring
    BIntMod(), // TODO: MontgomeryBigint<BigUint> or MontgomeryInt<BigUint>? the latter one could make the struct sendable
    /// Quadratic number
    Quad(Quadratic64),
    /// Multi-precision Quadratic number
    BQuad(BigQuadratic),
    /// Quadratic integer
    QuadInt(), // TODO: Quadratic Integer
    /// Multi-precision Quadratic INteger
    BQuadInt(),
}

// These converters will automatically downcast to smaller representation if possible.

impl From<i64> for Subject {
    fn from(v: i64) -> Self {
        Subject::Int(v)
    }
}
impl From<u64> for Subject {
    fn from(v: u64) -> Self {
        match v.try_into() {
            Ok(u) => Subject::Int(u),
            Err(_) => Subject::BInt(BigInt::from_u64(v).unwrap()),
        }
    }
}
impl From<usize> for Subject {
    fn from(v: usize) -> Self {
        match v.try_into() {
            Ok(u) => Subject::Int(u),
            Err(_) => Subject::BInt(BigInt::from_usize(v).unwrap()),
        }
    }
}
impl From<f64> for Subject {
    fn from(v: f64) -> Self {
        if v.is_nan() {
            panic!("nan is not accepted as a suan::Subject")
        } else if v.is_infinite() {
            Subject::Infinity(v.is_sign_positive())
        } else {
            Subject::Real(v)
        }
    }
}
impl From<BigInt> for Subject {
    fn from(v: BigInt) -> Self {
        match v.to_i64() {
            Some(s) => Subject::Int(s),
            None => Subject::BInt(v),
        }
    }
}
impl From<BigUint> for Subject {
    fn from(v: BigUint) -> Self {
        match v.to_i64() {
            Some(s) => Subject::Int(s),
            None => Subject::BInt(BigInt::from(v)),
        }
    }
}
impl From<Rational64> for Subject {
    fn from(v: Rational64) -> Self {
        if v.is_integer() {
            Subject::Int(v.to_integer())
        } else {
            Subject::Rational(v)
        }
    }
}
impl From<BigRational> for Subject {
    fn from(v: BigRational) -> Self {
        if v.is_integer() {
            v.to_integer().into()
        } else {
            match (v.numer().to_i64(), v.denom().to_i64()) {
                (Some(n), Some(d)) => Subject::Rational(Rational64::new(n, d)),
                _ => Subject::BRational(v),
            }
        }
    }
}
impl From<Quadratic64> for Subject {
    fn from(v: Quadratic64) -> Self {
        if v.is_integer() {
            Subject::Int(v.to_integer().value())
        } else if v.is_rational() {
            Subject::Rational(v.to_rational().value())
        } else {
            Subject::Quad(v)
        }
    }
}
impl From<BigQuadratic> for Subject {
    fn from(v: BigQuadratic) -> Self {
        let (ra, rb, rc, rr) = v.parts();
        match (ra.to_i64(), rb.to_i64(), rc.to_i64(), rr.to_i64()) {
            (Some(sa), Some(sb), Some(sc), Some(sr)) => {
                Subject::Quad(Quadratic64::new(sa, sb, sc, sr))
            }
            _ => Subject::BQuad(v),
        }
    }
}
impl From<Complex64> for Subject {
    fn from(v: Complex64) -> Self {
        Subject::Complex(v)
    }
}
