use crate::subject::Subject::{self, *};
use num_bigint::BigInt;
use num_irrational::Quadratic64;
use num_traits::{One, Zero};
use std::fmt::{Debug, Display, Formatter};

// short print int to first 8 digits
fn sprinti(f: &mut Formatter<'_>, i: &i64) -> std::fmt::Result {
    if i.abs() >= 100000000 {
        write!(f, "{:.7E}", i)
    } else {
        write!(f, "{}", i)
    }
}

// short print bigint to first 8 digits
fn sprintbi(f: &mut Formatter<'_>, i: &BigInt) -> std::fmt::Result {
    // TODO: use standard formatting after https://github.com/rust-num/num-bigint/pull/214 is merged
    let decimal_digits = (i.bits() as f64) / 10f64.log2();
    if decimal_digits > 7. {
        let digits = decimal_digits.floor() as u32;
        write!(
            f,
            "{}E{}",
            i / BigInt::from(10u8).pow(digits - 7),
            digits - 7
        )
    } else {
        write!(f, "{}", i)
    }
}

// short print float to first 8 digits
fn sprintf(f: &mut Formatter<'_>, v: &f64) -> std::fmt::Result {
    if v.abs() >= 1e8 {
        write!(f, "{:.7E}", v)
    } else if v.abs() <= 1e-4 {
        write!(f, "{:.7E}", v)
    } else {
        let s = v.to_string();
        if s.len() > 9 {
            f.write_str(&s[..9])
        } else {
            f.write_str(&s)
        }
    }
}

impl Debug for Subject {
    fn fmt(&self, f: &mut Formatter<'_>) -> std::fmt::Result {
        match self {
            Infinity(v) => {
                if *v {
                    write!(f, "INF+")
                } else {
                    write!(f, "INF-")
                }
            }
            Int(v) => {
                write!(f, "INT ")?;
                sprinti(f, v)
            }
            BInt(v) => {
                write!(f, "INT* ")?;
                sprintbi(f, v)
            }
            Real(v) => {
                write!(f, "REAL ")?;
                sprintf(f, v)
            }
            Rational(v) => {
                if v.denom().is_one() {
                    write!(f, "RATIONAL ")?;
                    sprinti(f, v.numer())
                } else {
                    write!(f, "RATIONAL ")?;
                    sprinti(f, v.numer())?;
                    write!(f, " / ")?;
                    sprinti(f, v.denom())
                }
            }
            Complex(v) => match (v.re == 0., v.im == 0.) {
                (true, true) => write!(f, "COMPLEX 0"),
                (true, false) => {
                    write!(f, "COMPLEX ")?;
                    sprintf(f, &v.im)?;
                    write!(f, "i")
                }
                (false, true) => {
                    write!(f, "COMPLEX ")?;
                    sprintf(f, &v.re)
                }
                (false, false) => {
                    write!(f, "COMPLEX ")?;
                    sprintf(f, &v.re)?;
                    if v.im > 0. {
                        write!(f, " + ")?;
                        if v.im != 1. {
                            sprintf(f, &v.im)?
                        }
                    } else {
                        write!(f, " - ")?;
                        if v.im != -1. {
                            sprintf(f, &-v.im)?
                        }
                    }
                    write!(f, "i")
                }
            },
            Quad(v) => {
                let (a, b, c, r) = v.parts();
                write!(f, "QUAD ")?;

                match (a.is_zero(), b.is_zero()) {
                    (true, true) => write!(f, "0"),
                    (false, false) => {
                        // numerator
                        write!(f, "(")?;
                        sprinti(f, a)?;

                        if b > &0 {
                            write!(f, " + ")?;
                            sprinti(f, b)?;
                        } else {
                            write!(f, " - ")?;
                            sprinti(f, &-*b)?;
                        }
                        write!(f, "√")?;
                        sprinti(f, r)?;

                        // denumerator
                        if c.is_one() {
                            Ok(())
                        } else {
                            write!(f, " / ")?;
                            sprinti(f, c)
                        }
                    },
                    (az, _) => {
                        // numerator
                        if az {
                            match b {
                                1 => {},
                                -1 => write!(f, "-")?,
                                b => sprinti(f, b)?
                            }
                            write!(f, "√")?;
                            sprinti(f, r)?;
                        } else {
                            sprinti(f, a)?;
                        }

                        // denumerator
                        if c.is_one() {
                            Ok(())
                        } else {
                            write!(f, " / ")?;
                            sprinti(f, c)
                        }
                    },
                }
            }
            _ => {
                // XXX: currently delegate to Display, should implement with sprint functions
                Display::fmt(self, f)
            }
        }
    }
}

impl Display for Subject {
    fn fmt(&self, f: &mut Formatter<'_>) -> std::fmt::Result {
        match self {
            Infinity(v) => {
                if *v {
                    write!(f, "+inf")
                } else {
                    write!(f, "-inf")
                }
            }
            Int(v) => Display::fmt(v, f),
            BInt(v) => Display::fmt(v, f),
            Real(v) => Display::fmt(v, f),
            Rational(v) => Display::fmt(v, f),
            BRational(v) => Display::fmt(v, f),
            Quad(v) => Display::fmt(v, f),
            BQuad(v) => Display::fmt(v, f),
            Complex(v) => Display::fmt(v, f),
            _ => write!(f, "..."),
        }
    }
}

pub enum Rounding {
    Nearest,
    Up,
    Down,
    ToZero,
    AwayZero
}

pub enum Precision {
    /// Precision in binary bits
    Bits(usize),
    /// Precision in decimal digits
    Digits(usize)
}

// conversions, precision will ensure the result is exact to at least the given position
impl Subject {
    pub fn int(self, rounding: ()) {} // TODO: default rounding mode is nearest.
    pub fn real(self, precision: (), rounding: ()) {} // TODO: default precision from integer is f64
    pub fn rational(self, precision: (), rounding: ()) {} // TODO: precision of rational is the scale of the denominator
    pub fn complex(self, precision: (), rounding: ()) {} // TODO: panics when convert from complex to normal number
    pub fn intmod(self, modulus: (), rounding: ()) {}
    pub fn quad(self, precision: (), rounding: ()) {}
    pub fn quadint(self, base: (), rounding: ()) {}
}

impl Subject {
    /// Get the precision of the number, the return variant depends on the underlying implementation
    pub fn prec(&self) -> Precision { unimplemented!() }
    /// Get the integral part of the number
    pub fn trunc(&self) -> Subject { unimplemented!() }
    /// Get the fractional part of the number
    pub fn frac(&self) -> Subject { unimplemented!() }
}

fn abs() {}
fn conj() {}
fn arg() {}
