//! Implementation of power and logarithm functions:
//!
//! - Power and roots: sqrt, cbrt, root, pow, log (ln, log2, log10), exp, hypot
//! - Floored power and roots: sqrt_, cbrt_, root_, log_
//! - More precise variants: expm1, powm1, ln1p, log1p
//!
//! Note that nth_root can be achieved by powering with a rational exponent
//!

use crate::common::*;
use crate::subject::Subject;
use num_complex::Complex64;
use num_integer::Roots;
use num_rational::{BigRational};
use num_irrational::{BigQuadratic, FromSqrt, Quadratic64};
use num_prime::ExactRoots;
use num_traits::{Signed, ToPrimitive};

pub fn sqrt(s: Subject) -> Result<Subject, SuErr> {
    match s {
        Subject::Int(v) => Ok(if v < 0 {
            Subject::Quad(Quadratic64::from_sqrt(v).unwrap())
        } else {
            if let Some(root) = (v as u64).sqrt_exact() {
                Subject::Int(root as i64)
            } else {
                Subject::Quad(Quadratic64::from_sqrt(v).unwrap())
            }
        }),
        Subject::BInt(v) => Ok(if v.is_negative() {
            Subject::from(BigQuadratic::from_sqrt(v).unwrap())
        } else {
            if let Some(root) = v.sqrt_exact() {
                Subject::from(root)
            } else {
                Subject::from(BigQuadratic::from_sqrt(v).unwrap())
            }
        }),
        Subject::Real(v) => Ok(Subject::Real(v.sqrt())),
        Subject::Rational(v) => Ok(
            match Quadratic64::from_sqrt(v) {
                Ok(sq) => sq.into(),
                Err(e) => {
                    let (num, den) = e.data.into();
                    BigQuadratic::from_sqrt(BigRational::new(num.into(), den.into())).unwrap().into()
                }
            }
        ),
        Subject::Quad(v) => match Quadratic64::from_sqrt(v) {
            Ok(sqrt) => Ok(Subject::Quad(sqrt)),
            Err(e) => match e.data.to_f64() {
                Some(f) => Ok(if f >= 0. {
                    Subject::Real(f.sqrt())
                } else {
                    Subject::Complex(Complex64::from(f).sqrt())
                }),
                None => Err(SuErr::ExceedMaxPrecision),
            },
        },
        Subject::BQuad(v) => match BigQuadratic::from_sqrt(v) {
            Ok(sqrt) => Ok(Subject::from(sqrt)),
            Err(e) => match e.data.to_f64() {
                Some(f) => Ok(if f >= 0. {
                    Subject::Real(f.sqrt())
                } else {
                    Subject::Complex(Complex64::from(f).sqrt())
                }),
                None => Err(SuErr::ExceedMaxPrecision),
            },
        },
        Subject::Complex(v) => Ok(Subject::Complex(v.sqrt())),
        _ => Err(SuErr::UnsupportedVariant),
    }
}

pub fn sqrt_(s: Subject) -> Result<Subject, SuErr> {
    match s {
        Subject::Int(v) => Ok(Subject::Int(v.sqrt())),
        Subject::BInt(v) => Ok(Subject::from(v.sqrt())),
        _ => Err(SuErr::UnsupportedVariant),
    }
}

pub fn cbrt(s: Subject) -> Result<Subject, SuErr> {
    match s {
        Subject::Int(v) => Ok(if let Some(root) = v.cbrt_exact() {
            Subject::Int(root)
        } else {
            Subject::Real((v as f64).cbrt())
        }),
        Subject::BInt(v) => {
            if let Some(root) = v.cbrt_exact() {
                Ok(Subject::from(root))
            } else {
                if let Some(f) = v.to_f64() {
                    Ok(Subject::Real(f.cbrt()))
                } else {
                    Err(SuErr::ExceedMaxPrecision)
                }
            }
        }
        Subject::Real(v) => Ok(Subject::Real(v.cbrt())),
        Subject::Complex(v) => Ok(Subject::Complex(v.cbrt())),
        _ => Err(SuErr::UnsupportedVariant),
    }
}

pub fn cbrt_(s: Subject) -> Result<Subject, SuErr> {
    match s {
        Subject::Int(v) => Ok(Subject::Int(v.cbrt())),
        Subject::BInt(v) => Ok(Subject::from(v.cbrt())),
        _ => Err(SuErr::UnsupportedVariant),
    }
}
