//! Implementation of unary and binary operators.

// Full equality and total ordering is implemented to support sorting with following strategy
// 1. Lexicographic ordering is used between Complex numbers
// 2. All numbers are (virtually) converted to a complex number when comparing
// But other binary operators will panic if two operands are not compatible

use num_bigint::BigInt;
use num_complex::Complex64;
use num_rational::{BigRational, Rational64};
use num_traits::float::FloatCore;
use num_traits::ToPrimitive;

use crate::subject::Subject::{self, *};
use std::cmp::Ordering;
use std::ops::{Add, Div};

impl PartialEq for Subject {
    fn eq(&self, other: &Self) -> bool {
        self.cmp(other).is_eq()
    }
}

impl Eq for Subject {}

impl PartialOrd for Subject {
    #[inline]
    fn partial_cmp(&self, other: &Self) -> Option<Ordering> {
        Some(self.cmp(other))
    }
}

fn cmp_i_f(lhs: &i64, rhs: &f64) -> Ordering {
    let lg2 = 64 - lhs.abs().leading_zeros();
    if lg2 <= 48 {
        // threshold from CPython
        (*lhs as f64).partial_cmp(rhs).unwrap() // lossless conversion
    } else {
        let (mantissa, exp, sign) = rhs.integer_decode();
        let mantissa = (sign as i64) * (mantissa as i64);
        if exp >= 0 {
            if let Some(shifted) = mantissa.checked_mul(1 << exp) {
                lhs.cmp(&shifted)
            } else {
                if mantissa > 0 {
                    Ordering::Less
                } else {
                    Ordering::Greater
                }
            }
        } else {
            if let Some(shifted) = lhs.checked_mul(1 << -exp) {
                shifted.cmp(&mantissa)
            } else {
                if lhs > &0 {
                    Ordering::Greater
                } else {
                    Ordering::Less
                }
            }
        }
    }
}

#[inline]
fn cmp_i_c(lhs: &i64, rhs: &Complex64) -> Ordering {
    match cmp_i_f(lhs, &rhs.re) {
        Ordering::Equal => 0f64.partial_cmp(&rhs.im).unwrap(),
        Ordering::Greater => Ordering::Greater,
        Ordering::Less => Ordering::Less,
    }
}

impl Ord for Subject {
    #[inline]
    fn cmp(&self, other: &Self) -> Ordering {
        match (self, other) {
            // comparison with infinity
            (Infinity(true), Infinity(true)) => Ordering::Equal,
            (Infinity(true), Infinity(false)) => Ordering::Greater,
            (Infinity(false), Infinity(true)) => Ordering::Less,
            (Infinity(false), Infinity(false)) => Ordering::Equal,
            (Infinity(true), _) => Ordering::Greater,
            (Infinity(false), _) => Ordering::Less,
            (_, Infinity(true)) => Ordering::Less,
            (_, Infinity(false)) => Ordering::Greater,

            // comparison with concrete types
            (Int(l), Int(r)) => l.cmp(r),
            (Int(l), BInt(r)) => BigInt::from(*l).cmp(r),
            (Int(l), Real(r)) => cmp_i_f(l, r),
            (Int(l), Complex(r)) => cmp_i_c(l, r),
            (BInt(l), BInt(r)) => l.cmp(r),
            (BInt(l), Int(r)) => l.cmp(&BigInt::from(*r)),
            (Real(l), Int(r)) => cmp_i_f(r, l).reverse(),
            (Real(l), Real(r)) => l.partial_cmp(r).unwrap(),
            _ => unimplemented!(),
        }
    }
}

impl Add for Subject {
    type Output = Self;
    #[inline]
    fn add(self, rhs: Self) -> Self::Output {
        match (self, rhs) {
            (_, Infinity(_)) | (Infinity(_), _) => panic!("operation with infinity is prohibited!"),
            (Int(l), Int(r)) => (l + r).into(),
            (Int(l), BInt(r)) => (l + r).into(),
            (Int(l), Real(r)) => (l as f64 + r).into(),
            (Int(l), Complex(r)) => (l as f64 + r).into(),
            (BInt(l), Int(r)) => (l + r).into(),
            (BInt(l), BInt(r)) => (l + r).into(),
            (BInt(l), Real(r)) => (l.to_f64().unwrap() + r).into(),
            (BInt(l), Complex(r)) => (l.to_f64().unwrap() + r).into(),
            (Real(l), Int(r)) => (l + r as f64).into(),
            (Real(l), BInt(r)) => (l + r.to_f64().unwrap()).into(),
            (Real(l), Real(r)) => (l + r).into(),
            (Real(l), Complex(r)) => (l + r).into(),
            (Complex(l), Int(r)) => (l + r as f64).into(),
            (Complex(l), BInt(r)) => (l + r.to_f64().unwrap()).into(),
            (Complex(l), Real(r)) => (l + r).into(),
            (Complex(l), Complex(r)) => (l + r).into(),
            _ => unimplemented!(),
        }
    }
}

impl Div for Subject {
    type Output = Self;
    #[inline]
    fn div(self, rhs: Self) -> Self::Output {
        match (self, rhs) {
            (Int(l), Int(r)) => Rational64::new(l, r).into(),
            (Int(l), BInt(r)) => BigRational::new(l.into(), r.clone()).into(),
            (Int(l), Real(r)) => (l as f64 / r).into(),
            (Int(l), Complex(r)) => (l as f64 / r).into(),
            (BInt(l), Int(r)) => BigRational::new(l.clone(), r.into()).into(),
            (BInt(l), BInt(r)) => BigRational::new(l.clone(), r.clone()).into(),
            (BInt(l), Real(r)) => (l.to_f64().unwrap() / r).into(),
            (BInt(l), Complex(r)) => (l.to_f64().unwrap() / r).into(),
            (Real(l), Int(r)) => (l / r as f64).into(),
            (Real(l), BInt(r)) => (l / r.to_f64().unwrap()).into(),
            (Real(l), Real(r)) => (l / r).into(),
            (Real(l), Complex(r)) => (l.to_f64().unwrap() / r).into(),
            (Complex(l), Int(r)) => (l / r as f64).into(),
            (Complex(l), BInt(r)) => (l / r.to_f64().unwrap()).into(),
            (Complex(l), Real(r)) => (l / r).into(),
            (Complex(l), Complex(r)) => (l / r).into(),
            _ => unimplemented!(),
        }
    }
}
