//! Number-theoretic functions:
//!
//! - Prime indexing: primes, nprimes, nth_prime (est+bounds), prime_pi (est+bounds)
//! - Primality: is_prime, is_prp, is_sprp, is_lprp, is_slprp, ..
//! - Factorization: factors, divisor, pollard_rho, pollard_brent, ..
//! - General: legendre, jacobi, kronecker, moebius, ..
//!

use crate::common::*;
use crate::subject::Subject;
use either::Either;
use lazy_static::lazy_static;
use num_prime::{
    buffer::{NaiveBuffer, PrimeBufferExt},
    nt_funcs, FactorizationConfig, Primality, PrimalityTestConfig,
};
use std::collections::BTreeMap;

lazy_static! {
    static ref PRIMEBUF: NaiveBuffer = NaiveBuffer::new();
}

/// Test primality of the number. Only applicable to integers, and negative integers will
/// be tested with its absolute value.
pub fn is_prime(s: Subject, config: Option<PrimalityTestConfig>) -> Result<Primality, SuErr> {
    match s {
        Subject::Int(v) => Ok(match nt_funcs::is_prime64(v.abs() as u64) {
            true => Primality::Yes,
            false => Primality::No,
        }),
        Subject::BInt(v) => Ok(PRIMEBUF.is_prime(&v.into_parts().1, config)),
        _ => Err(SuErr::UnsupportedVariant),
    }
}

/// Number factorization. Only applicable to integers, and negative integers will
/// be factored with its absolute value.
pub fn factors(
    s: Subject,
    config: Option<FactorizationConfig>,
) -> Result<Either<BTreeMap<Subject, Subject>, Vec<Subject>>, SuErr> {
    match s {
        Subject::Int(v) => {
            let result = nt_funcs::factors64(v.abs() as u64);
            Ok(Either::Left(
                result
                    .into_iter()
                    .map(|(k, v)| (k.into(), v.into()))
                    .collect(),
            ))
        }
        Subject::BInt(v) => match PRIMEBUF.factors(v.into_parts().1, config) {
            Ok(result) => Ok(Either::Left(
                result
                    .into_iter()
                    .map(|(k, v)| (k.into(), v.into()))
                    .collect(),
            )),
            Err(result) => Ok(Either::Right(
                result.into_iter().map(|v| v.into()).collect(),
            )),
        },
        _ => Err(SuErr::UnsupportedVariant),
    }
}
