//! Implementation of power and logarithm functions:
//!
//! - Power and roots: sqrt, cbrt, root, pow, log (ln, log2, log10), exp, hypot
//! - Floored power and roots: sqrt_, cbrt_, root_, log_
//! - More precise variants: expm1, powm1, ln1p, log1p
//!
//! Note that nth_root can be achieved by powering with a rational exponent
//!

use crate::common::*;
use crate::subject::Subject;
use num_complex::Complex64;
use num_integer::Roots;
use num_irrational::traits::FromSqrt;
use num_irrational::QuadraticSurd;
use num_prime::ExactRoots;
use num_traits::ToPrimitive;
use std::rc::Rc;

pub fn sqrt(s: Subject) -> Result<Subject, SuErr> {
    match s {
        Subject::Int(v) => Ok(if v < 0 {
            // TODO: return complex quad number instead
            Subject::Complex(Complex64::new(0., (-v as f64).sqrt()))
        } else {
            if let Some(root) = (v as u64).sqrt_exact() {
                Subject::Int(root as i64)
            } else {
                Subject::Real((v as f64).sqrt())
            }
        }),
        Subject::BInt(v) => {
            // TODO: implement same logic as Int
            if let Some(f) = v.to_f64() {
                Ok(Subject::Real(f.sqrt()))
            } else {
                Err(SuErr::ExceedMaxPrecision)
            }
        }
        Subject::Real(v) => Ok(Subject::Real(v.sqrt())),
        Subject::Quad(v) => match QuadraticSurd::from_sqrt(v) {
            Ok(sqrt) => Ok(Subject::Quad(sqrt)),
            Err(_) => {
                // TODO: convert to float and then sqrt
                unimplemented!()
            }
        },
        _ => Err(SuErr::UnsupportedVariant)
    }
}

pub fn sqrt_(s: Subject) -> Result<Subject, SuErr> {
    match s {
        Subject::Int(v) => Ok(Subject::Int(v.sqrt())),
        Subject::BInt(v) => {
            let root = v.sqrt();
            if let Some(v64) = root.to_i64() {
                Ok(Subject::Int(v64))
            } else {
                Ok(Subject::BInt(Rc::new(root)))
            }
        }
        _ => Err(SuErr::UnsupportedVariant)
    }
}

pub fn cbrt(s: Subject) -> Result<Subject, SuErr> {
    match s {
        Subject::Int(v) => Ok(if let Some(root) = (v as u64).cbrt_exact() {
            Subject::Int(root as i64)
        } else {
            Subject::Real((v as f64).cbrt())
        }),
        Subject::BInt(v) => {
            if let Some(f) = v.to_f64() {
                Ok(Subject::Real(f.cbrt()))
            } else {
                Err(SuErr::ExceedMaxPrecision)
            }
        }
        Subject::Real(v) => Ok(Subject::Real(v.cbrt())),
        _ => Err(SuErr::UnsupportedVariant)
    }
}

pub fn cbrt_(s: Subject) -> Result<Subject, SuErr> {
    match s {
        Subject::Int(v) => Ok(Subject::Int(v.cbrt())),
        Subject::BInt(v) => {
            let root = v.cbrt();
            if let Some(v64) = root.to_i64() {
                Ok(Subject::Int(v64))
            } else {
                Ok(Subject::BInt(Rc::new(root)))
            }
        }
        _ => Err(SuErr::UnsupportedVariant)
    }
}
