use regex::Regex;
use std::collections::HashMap;
pub fn all_elements_equal<I: Iterator, T>(iter: &mut I) -> bool
where
    T: PartialEq<T>,
    I: Iterator<Item = T>,
{
    let first_elem = match iter.next() {
        Some(first_elem) => first_elem,
        None => return true,
    };
    iter.all(|elem| elem == first_elem)
}

pub fn is_hashmap_dataframe(map: &mut HashMap<String, Vec<String>>) -> bool {
    map.len() <= 1 || all_elements_equal(&mut map.iter_mut().map(|elem| elem.1.len()))
}

pub fn csv_header(columns: Vec<String>) -> String {
    remove_trailing_comma(
        columns
            .iter()
            .map(|column| column.to_owned() + ",")
            .collect::<String>()
            + "\n",
    )
}

pub fn remove_trailing_comma(string: String) -> String {
    Regex::new(r",\n$")
        .unwrap()
        .replace(&Regex::new(r",$").unwrap().replace(&string, ""), "\n")
        .to_string()
}

#[cfg(test)]
mod tests {
    use super::*;
    mod test_all_elements_equal {
        use super::*;
        #[test]
        fn int_vec_equal() {
            assert_eq!(all_elements_equal(&mut vec![1, 1, 1].into_iter()), true)
        }
        #[test]
        fn char_vec_equal() {
            assert_eq!(
                all_elements_equal(&mut vec!["a", "a", "a"].into_iter()),
                true
            )
        }
        #[test]
        fn string_vec_equal() {
            assert_eq!(
                all_elements_equal(&mut vec!["abc", "abc", "abc"].into_iter()),
                true
            )
        }
        #[test]
        fn int_vec_not_equal() {
            assert_eq!(all_elements_equal(&mut vec![1, 1, 2].into_iter()), false)
        }
    }
    mod test_is_hashmap_dataframe {
        use super::*;
        #[test]
        fn valid_hashmap() {
            let mut test_hashmap = HashMap::new();
            test_hashmap.insert(
                String::from("test-column-0"),
                vec![String::from("value-0-0"), String::from("value-0-1")],
            );
            test_hashmap.insert(
                String::from("test-column-1"),
                vec![String::from("value-1-0"), String::from("value-1-1")],
            );
            assert_eq!(is_hashmap_dataframe(&mut test_hashmap), true);
        }
        #[test]
        fn invalid_hashmap() {
            let mut test_hashmap = HashMap::new();
            test_hashmap.insert(
                String::from("test-column-0"),
                vec![String::from("value-0-0"), String::from("value-0-1")],
            );
            test_hashmap.insert(
                String::from("test-column-1"),
                vec![
                    String::from("value-1-0"),
                    String::from("value-1-1"),
                    String::from("value-1-2"),
                ],
            );
            assert_eq!(is_hashmap_dataframe(&mut test_hashmap), false);
        }
    }
    mod test_csv_header {
        use super::*;
        #[test]
        fn simple_test() {
            assert_eq!(
                "column-1,column-2,column-3\n",
                csv_header(vec![
                    String::from("column-1"),
                    String::from("column-2"),
                    String::from("column-3")
                ])
            )
        }
        #[test]
        fn one_column() {
            assert_eq!("column-1\n", csv_header(vec![String::from("column-1"),]))
        }
        #[test]
        fn no_data() {
            let empty_vector: Vec<String> = vec![];
            assert_eq!("\n", csv_header(empty_vector))
        }
    }
    mod test_remove_trailing_comma {
        use super::*;
        #[test]
        fn comma_before_eol() {
            assert_eq!(
                String::from("val-1 val-2 val-3\n"),
                remove_trailing_comma(String::from("val-1 val-2 val-3,\n"))
            )
        }
        #[test]
        fn comma() {
            assert_eq!(
                String::from("val-1 val-2 val-3"),
                remove_trailing_comma(String::from("val-1 val-2 val-3,"))
            )
        }
        #[test]
        fn irrelevant_comma() {
            assert_eq!(
                String::from("val-1, val-2 val-3"),
                remove_trailing_comma(String::from("val-1, val-2 val-3,"))
            )
        }
        #[test]
        fn empty_data() {
            assert_eq!(String::from(""), remove_trailing_comma(String::from("")))
        }
    }
}
