use crate::csv_data_frame::{CSVDataFrame, DataFrameErrors};
use crate::data_frame_list::DataFrameList;
use parquet::file::reader::{FileReader, SerializedFileReader};
use parquet::record::RowAccessor;
use parquet::schema::types::Type as ParquetSchemaType;
use std::convert::TryFrom;
use std::{fs::File, path::Path};

enum TransformError {
    CouldNotParseToString,
}

pub fn read_parquet(parquet_path: &str) -> Result<CSVDataFrame, DataFrameErrors> {
    let path = Path::new(parquet_path);
    let mut all_dataframes = DataFrameList::new();
    if let Ok(file) = File::open(&path) {
        let reader = SerializedFileReader::new(file).unwrap();
        let schema = reader.metadata().file_metadata().schema();
        for column in schema.get_fields().iter() {
            let mut relevant_fields = reader
                .metadata()
                .file_metadata()
                .schema()
                .get_fields()
                .to_vec();

            relevant_fields.retain(|elem| elem.name() == column.name());

            let schema_projection = ParquetSchemaType::group_type_builder("schema")
                .with_fields(&mut relevant_fields)
                .build()
                .unwrap();
            if let Err(DataFrameErrors::DifferrentNumberRows) = all_dataframes.push(
                get_records_from_schema_projection(parquet_path, schema_projection, column.name()),
            ) {
                return Err(DataFrameErrors::DifferrentNumberRows);
            }
        }
        return match CSVDataFrame::try_from(all_dataframes) {
            Ok(dataframe) => Ok(dataframe),
            Err(_) => Err(DataFrameErrors::DifferrentNumberRows),
        };
    } else {
        Err(DataFrameErrors::CouldNotOpenFile)
    }
}

fn get_records_from_schema_projection(
    path: &str,
    schema: ParquetSchemaType,
    column_name: &str,
) -> CSVDataFrame {
    if let Ok(file) = File::open(&Path::new(path)) {
        let reader = SerializedFileReader::new(file).unwrap();
        get_csv_dataframe_from_row_iter(reader.get_row_iter(Some(schema)).unwrap(), column_name)
    } else {
        panic!("Could not open file {}", path);
    }
}

fn get_csv_dataframe_from_row_iter(
    row_iter: parquet::record::reader::RowIter,
    column_name: &str,
) -> CSVDataFrame {
    let mut values = Vec::new();

    for row in row_iter {
        if let Ok(value) = get_elem(row, 0) {
            values.push(value)
        }
    }
    CSVDataFrame::from([(String::from(column_name), values)])
}
fn get_elem(row: parquet::record::Row, idx: usize) -> Result<String, TransformError> {
    if let Ok(value) = row.get_bool(idx) {
        return Ok(format!("{}", value));
    }
    if let Ok(value) = row.get_byte(idx) {
        return Ok(format!("{}", value));
    }
    if let Ok(value) = row.get_short(idx) {
        return Ok(format!("{}", value));
    }
    if let Ok(value) = row.get_int(idx) {
        return Ok(format!("{}", value));
    }
    if let Ok(value) = row.get_long(idx) {
        return Ok(format!("{}", value));
    }
    if let Ok(value) = row.get_ubyte(idx) {
        return Ok(format!("{}", value));
    }
    if let Ok(value) = row.get_ushort(idx) {
        return Ok(format!("{}", value));
    }
    if let Ok(value) = row.get_uint(idx) {
        return Ok(format!("{}", value));
    }
    if let Ok(value) = row.get_string(idx) {
        return Ok(value.to_string());
    }
    if let Ok(value) = row.get_float(idx) {
        return Ok(format!("{}", value));
    }
    if let Ok(value) = row.get_double(idx) {
        return Ok(format!("{}", value));
    }
    if let Ok(value) = row.get_decimal(idx) {
        return Ok(format!("{:?}", value));
    }
    Err(TransformError::CouldNotParseToString)
}

// #[cfg(test)]
// mod test {
//     use super::*;
//     mod test_get_elem {
//         use super::*;
//         #[test]
//         fn test_bool() {
//             let fields = vec![(String::from("column-name"), Field::Bool(true))];
//             let test = Row { fields };
//         }
//     }
// }
