use crate::csv_data_frame::{CSVDataFrame, DataFrameErrors};
#[allow(unused_imports)]
use std::collections::HashMap;
#[allow(unused_imports)]
use std::convert::TryFrom;
use std::slice::Iter;
pub struct DataFrameList {
    data_frames: Vec<CSVDataFrame>,
}

impl DataFrameList {
    pub fn new() -> Self {
        DataFrameList {
            data_frames: Vec::new(),
        }
    }
    pub fn push(&mut self, data_frame: CSVDataFrame) -> Result<(), DataFrameErrors> {
        if self.data_frames.is_empty() || data_frame.n_rows() == self.data_frames[0].n_rows() {
            self.data_frames.push(data_frame);
            Ok(())
        } else {
            Err(DataFrameErrors::DifferrentNumberRows)
        }
    }
    pub fn len(&self) -> usize {
        self.data_frames.len()
    }
    pub fn is_empty(&self) -> bool {
        self.len() == 0
    }
    pub fn iter(&self) -> Iter<'_, CSVDataFrame> {
        self.data_frames.iter()
    }
}

impl Default for DataFrameList {
    fn default() -> Self {
        Self::new()
    }
}

#[cfg(test)]
mod test {
    use super::*;
    #[test]
    fn test_push_different_rows() {
        // We add two Dataframes with one column each and test that
        // DataFrameList correctly reports 2 as `len`.
        let mut test_list = DataFrameList::new();
        let _ = test_list.push(
            CSVDataFrame::try_from(HashMap::from([(
                String::from("column-1"),
                vec![String::from("a"), String::from("b"), String::from("c")],
            )]))
            .unwrap(),
        );
        let failed = test_list.push(
            CSVDataFrame::try_from(HashMap::from([(
                String::from("column-1"),
                vec![String::from("a"), String::from("b")],
            )]))
            .unwrap(),
        );
        // Test that the `push`-method gave back an error.
        assert!(matches!(
            failed.err().unwrap(),
            DataFrameErrors::DifferrentNumberRows
        ));
        // Make sure the additional DataFrame was not stored
        // internally
        assert_eq!(test_list.len(), 1);
    }
    #[test]
    fn test_len() {
        // We add two Dataframes with one column each and test that
        // DataFrameList correctly reports 2 as `len`.
        let mut test_list = DataFrameList::new();
        let _ = test_list.push(
            CSVDataFrame::try_from(HashMap::from([(
                String::from("column-1"),
                vec![String::from("a"), String::from("b"), String::from("c")],
            )]))
            .unwrap(),
        );
        let _ = test_list.push(
            CSVDataFrame::try_from(HashMap::from([(
                String::from("column-1"),
                vec![String::from("a"), String::from("b"), String::from("c")],
            )]))
            .unwrap(),
        );
        assert_eq!(test_list.len(), 2);
    }
}
