from collections import OrderedDict

lifetimes = ["<'a>"]


def strip_lifetimes(element):
    for i in lifetimes:
        if element.endswith(i):
            element = element.removesuffix(i)
            break

    return element


def to_camel_case(snake_str):
    components = snake_str.split('_')
    # We capitalize the first letter of each component except the first
    # one with the 'title' method and join them together.
    return components[0] + "".join(x.title() for x in components[1:])


def parse_function_signature(line):
    line = line.removeprefix("pub ")
    line = line.removeprefix("fn ")

    (function_name, remains) = line.split("(", 1)
    args = remains[:remains.rindex(")")]
    if args:
        args = args.split(", ")
    else:
        args = []

    ret_args = []

    for i in args:
        (arg_name, arg_type) = i.split(": ", 1)
        ret_args.append((arg_name, arg_type))

    return (function_name, ret_args)


def parse_function_signatures(path):
    ret = []

    with open(path, encoding="utf-8") as f:
        lines = [i.strip() for i in f.readlines()]

    for line in lines:
        if line.startswith("pub fn ") or line.startswith("fn "):
            ret.append(parse_function_signature(line))

    return ret


def parse_const(line):
    line = line.removeprefix("pub ")
    line = line.removeprefix("const ").split(": ")
    const_name = line[0]
    const_type = line[1].split(" = ")
    const_value = const_type[1].removesuffix(";")
    const_type = const_type[0]

    return (const_name, const_type, const_value)


def parse_types(path):
    with open(path, encoding="utf-8") as f:
        lines = [i.strip() for i in f.readlines()]

    types = {}

    for line in lines:
        if line.startswith("pub const "):
            const = parse_const(line)
            types[const[0]] = (const[1], const[2])

    found = False
    struct_name = None
    for line in lines:
        if found and not line.startswith("//") and not line.startswith("}"):
            element = line.removeprefix("pub ")
            element = element.removesuffix(",")
            element = strip_lifetimes(element)
            (name, typ) = element.split(": ", 1)
            types[struct_name].append((name, typ))

        if line.startswith("pub struct "):
            found = True
            struct_name = line.split()[2]

            struct_name = strip_lifetimes(struct_name)

            types[struct_name] = []

        if line.startswith("}"):
            found = False

    return types


def parse_accounts_struct(path):
    with open(path, encoding="utf-8") as f:
        lines = [i.strip() for i in f.readlines()]

    accounts = []

    found = False
    for line in lines:
        if found and not line.startswith("//") and not line.startswith("}"):
            element = line.removeprefix("pub ")
            name = element.split(": ")[0]
            attrs = element.split("// ")[1][1:-1]

            account["name"] = to_camel_case(name)
            account["isMut"] = False
            account["isSigner"] = False

            if "writable" in attrs:
                account["isMut"] = True

            if "signer" in attrs:
                account["isSigner"] = True

            accounts.append(account)
            account = OrderedDict()

        if line.startswith("pub struct "):
            found = True
            account = OrderedDict()

        if line.startswith("}"):
            found = False

    return accounts


def lookup_type_idl(n, types):
    if n == "u64":
        return "u64"

    if n == "bool":
        return "bool"

    if n.startswith("[u8; "):
        # This should be a fixed-size array
        size = n.split()[1][:-1]
        if size in types:
            return {"array": ["u8", types[size][1]]}
        if size.isnumeric():
            return {"array": ["u8", int(size)]}

        raise Exception("Couldn't parse fixed-size array")

    raise Exception(f"Could not find type {n}")


def lookup_type_borsh(n, t, types):
    if t == "u64":
        return f"['{to_camel_case(n)}', 'u64'],\n"

    if t == "bool" or t == "u8":
        return f"['{to_camel_case(n)}', 'u8'],\n"

    if t.startswith("[u8; "):
        # This should be a fixed-size array
        size = t.split()[1][:-1]
        if size in types:
            return f"['{to_camel_case(n)}', [{types[size][1]}]],\n"

    if t == "Pubkey":
        return f"['{to_camel_case(n)}', [32]],\n"

    if t == "f32":
        return f"['{to_camel_case(n)}', 'f32'],\n"

    if t == "Vec<u8>":
        if n == "ix_padding":
            size = int(types["CREATEPARAMS_PADDING"][1])
        else:
            raise Exception(f"Unknown Vec size for {n}")

        return f"['{to_camel_case(n)}', '[{size}]],\n"

    raise Exception(f"Could not find type {t}")


def expand_args_idl(arglist, types):
    ret = []

    for arg in arglist:
        argtype = arg[1]

        # TODO: Add more primitives when needed
        if argtype == "u64":
            a = OrderedDict()
            a["name"] = to_camel_case(arg[0])
            a["type"] = lookup_type_idl("u64", types)
            ret.append(a)
            continue

        if argtype in types:
            t = types[argtype]
            if not isinstance(t, list):
                raise Exception("This should be a list")

            for i in t:
                a = OrderedDict()
                a["name"] = to_camel_case(i[0])
                a["type"] = lookup_type_idl(i[1], types)
                ret.append(a)

            continue

        raise Exception("Unreachable")

    return ret
