use std::collections::HashMap;
use std::io::{BufRead, BufReader};
use std::os::unix::net::UnixStream;
use std::sync::{Arc, RwLock};
use serde::{Serialize};
use serde::de::DeserializeOwned;
use streamduck_daemon::socket::daemon_data::{AddDevice, AddDeviceResult, Device, ListDevices, GetDevice, GetDeviceResult, RemoveDevice, RemoveDeviceResult, SocketAPIVersion, ReloadDeviceConfigsResult, ReloadDeviceConfigResult, SaveDeviceConfigsResult, SaveDeviceConfigResult, SetBrightnessResult, ReloadDeviceConfig, SaveDeviceConfig, SetBrightness, ListModules, ListComponents};
use streamduck_daemon::socket::{parse_packet_to_data, send_no_data_packet_with_requester, send_packet_with_requester, SocketData, SocketPacket};
use streamduck_daemon::versions::SOCKET_API;
use crate::{ComponentDefinition, PluginMetadata, SDClient, SDClientError};

/// Definition of Unix Socket based client
pub struct UnixClient {
    connection: RwLock<BufReader<UnixStream>>
}

#[allow(dead_code)]
impl UnixClient {
    /// Initializes client using unix domain socket
    pub fn new() -> Result<Arc<Box<dyn SDClient>>, std::io::Error> {
        let client: Arc<Box<dyn SDClient>> = Arc::new(Box::new(UnixClient {
            connection: RwLock::new(BufReader::new(UnixStream::connect("/tmp/streamduck.sock")?))
        }));

        let daemon_version = client.version().expect("Failed to retrieve version");

        if daemon_version != SOCKET_API.1 {
            println!("[Warning] Version of client library doesn't match daemon API version. Client: {}, Daemon: {}", SOCKET_API.1, daemon_version);
        }

        Ok(client)
    }

    fn process_request<Req: SocketData + Serialize, Res: SocketData + DeserializeOwned>(&self, request: &Req) -> Result<Res, SDClientError> {
        let mut handle = self.connection.write().unwrap();

        send_packet_with_requester(handle.get_mut(), "", request)?;

        let mut line = String::new();
        handle.read_line(&mut line)?;

        let packet: SocketPacket = serde_json::from_str(&line)?;

        Ok(parse_packet_to_data(&packet)?)
    }

    fn process_request_without_data<Res: SocketData + DeserializeOwned>(&self) -> Result<Res, SDClientError> {
        let mut handle = self.connection.write().unwrap();

        send_no_data_packet_with_requester::<Res>(handle.get_mut(), "")?;

        let mut line = String::new();
        handle.read_line(&mut line)?;

        let packet: SocketPacket = serde_json::from_str(&line)?;

        Ok(parse_packet_to_data(&packet)?)
    }
}

impl SDClient for UnixClient {
    fn version(&self) -> Result<String, SDClientError> {
        let response: SocketAPIVersion = self.process_request_without_data()?;

        Ok(response.version)
    }

    fn device_list(&self) -> Result<Vec<Device>, SDClientError> {
        let response: ListDevices = self.process_request_without_data()?;

        Ok(response.devices)
    }

    fn get_device(&self, serial_number: &str) -> Result<GetDeviceResult, SDClientError> {
        let response: GetDeviceResult = self.process_request(&GetDevice {
            serial_number: serial_number.to_string()
        })?;

        Ok(response)
    }

    fn add_device(&self, serial_number: &str) -> Result<AddDeviceResult, SDClientError> {
        let response: AddDeviceResult = self.process_request(&AddDevice {
            serial_number: serial_number.to_string()
        })?;

        Ok(response)
    }

    fn remove_device(&self, serial_number: &str) -> Result<RemoveDeviceResult, SDClientError> {
        let response: RemoveDeviceResult = self.process_request(&RemoveDevice {
            serial_number: serial_number.to_string()
        })?;

        Ok(response)
    }

    fn reload_device_configs(&self) -> Result<ReloadDeviceConfigsResult, SDClientError> {
        let response: ReloadDeviceConfigsResult = self.process_request_without_data()?;

        Ok(response)
    }

    fn reload_device_config(&self, serial_number: &str) -> Result<ReloadDeviceConfigResult, SDClientError> {
        let response: ReloadDeviceConfigResult = self.process_request(&ReloadDeviceConfig {
            serial_number: serial_number.to_string()
        })?;

        Ok(response)
    }

    fn save_device_configs(&self) -> Result<SaveDeviceConfigsResult, SDClientError> {
        let response: SaveDeviceConfigsResult = self.process_request_without_data()?;

        Ok(response)
    }

    fn save_device_config(&self, serial_number: &str) -> Result<SaveDeviceConfigResult, SDClientError> {
        let response: SaveDeviceConfigResult = self.process_request(&SaveDeviceConfig {
            serial_number: serial_number.to_string()
        })?;

        Ok(response)
    }

    fn set_brightness(&self, serial_number: &str, brightness: u8) -> Result<SetBrightnessResult, SDClientError> {
        let response: SetBrightnessResult = self.process_request(&SetBrightness {
            serial_number: serial_number.to_string(),
            brightness
        })?;

        Ok(response)
    }

    fn list_modules(&self) -> Result<Vec<PluginMetadata>, SDClientError> {
        let response: ListModules = self.process_request_without_data()?;

        Ok(response.modules)
    }

    fn list_components(&self) -> Result<HashMap<String, HashMap<String, ComponentDefinition>>, SDClientError> {
        let response: ListComponents = self.process_request_without_data()?;

        Ok(response.components)
    }
}