use std::collections::HashMap;
use std::io::Error;
use streamduck_daemon::socket::daemon_data::{AddDeviceResult, Device, GetDeviceResult, ReloadDeviceConfigResult, ReloadDeviceConfigsResult, RemoveDeviceResult, SaveDeviceConfigResult, SaveDeviceConfigsResult, SetBrightnessResult};

pub use streamduck_daemon as daemon;
use streamduck_daemon::modules::components::ComponentDefinition;
use streamduck_daemon::modules::PluginMetadata;
use streamduck_daemon::socket::SocketError;

#[cfg(target_family = "unix")]
pub mod unix;

/// Trait that defines a client, implementations of clients must be separate due to conditional compiling
pub trait SDClient {
    // Version
    /// Retrieves version of the daemon socket API
    fn version(&self) -> Result<String, SDClientError>;

    // Device management
    /// Device list
    fn device_list(&self) -> Result<Vec<Device>, SDClientError>;
    /// Adds device to managed list
    fn get_device(&self, serial_number: &str) -> Result<GetDeviceResult, SDClientError>;
    /// Adds device to managed list
    fn add_device(&self, serial_number: &str) -> Result<AddDeviceResult, SDClientError>;
    /// Removes device from managed list
    fn remove_device(&self, serial_number: &str) -> Result<RemoveDeviceResult, SDClientError>;

    // Device configuration
    /// Reloads all device configs, all changes will be lost executing this
    fn reload_device_configs(&self) -> Result<ReloadDeviceConfigsResult, SDClientError>;
    /// Reloads device config for specific device, all changes will be lost executing this
    fn reload_device_config(&self, serial_number: &str) -> Result<ReloadDeviceConfigResult, SDClientError>;
    /// Saves all device configs
    fn save_device_configs(&self) -> Result<SaveDeviceConfigsResult, SDClientError>;
    /// Saves device config for specific device
    fn save_device_config(&self, serial_number: &str) -> Result<SaveDeviceConfigResult, SDClientError>;

    /// Sets device brightness, usually 0-100, but different for each device
    fn set_brightness(&self, serial_number: &str, brightness: u8) -> Result<SetBrightnessResult, SDClientError>;

    // Module management
    /// Lists all modules loaded by daemon
    fn list_modules(&self) -> Result<Vec<PluginMetadata>, SDClientError>;
    /// Lists all components that were introduced by modules
    fn list_components(&self) -> Result<HashMap<String, HashMap<String, ComponentDefinition>>, SDClientError>;
}

/// Errors that could happen with the client
#[derive(Debug)]
pub enum SDClientError {
    WriteError(std::io::Error),
    SerializeError(serde_json::Error),
    SocketError(streamduck_daemon::socket::SocketError),
    Custom(String)
}

impl From<std::io::Error> for SDClientError {
    fn from(err: Error) -> Self {
        SDClientError::WriteError(err)
    }
}

impl From<serde_json::Error> for SDClientError {
    fn from(err: serde_json::Error) -> Self {
        SDClientError::SerializeError(err)
    }
}

impl From<streamduck_daemon::socket::SocketError> for SDClientError {
    fn from(err: SocketError) -> Self {
        SDClientError::SocketError(err)
    }
}