use std::io::Write;
use std::str::Split;
use std::sync::Arc;
use streamduck_client::daemon::socket::daemon_data::{AddDeviceResult, DeviceType, GetDeviceResult, ReloadDeviceConfigResult, ReloadDeviceConfigsResult, RemoveDeviceResult, SaveDeviceConfigResult, SaveDeviceConfigsResult, SetBrightnessResult};
use streamduck_client::SDClient;
use crate::helps::COMMANDS;

type ClientRef<'a> = &'a Arc<Box<dyn SDClient>>;

pub fn prompt(client: Arc<Box<dyn SDClient>>) {
    println!("Streamduck CLI Prompt\n\nTo enter interactive UI mode, enter 'ui' command.\nTo view commands, enter 'help' command.\nTo exit, enter 'exit'.\n");


    let mut current_sn = String::new();

    loop {
        let mut line = String::new();
        print!("{}{}> ", current_sn, if current_sn.is_empty() { "" } else { " " });
        std::io::stdout().flush().unwrap();
        std::io::stdin().read_line(&mut line).unwrap();

        let mut args = line.trim().split(" ");

        if let Some(command) = args.next() {
            match command {
                "help" => println!("{}", prompt_help()),
                "exit" => break,

                "device" => {
                    if let Some(command) = args.next() {
                        match command {
                            "list" => device_list(&client),
                            "add" => add_device(&client, args),
                            "remove" => remove_device(&client, args),
                            _ => println!("device: Unknown command"),
                        }
                    } else {
                        println!("device: Unknown command");
                    }
                }

                "select" => {
                    if let Some(serial) = args.next() {
                        if let GetDeviceResult::Found(_) = client.get_device(serial).expect("Failed to check for device") {
                            current_sn = serial.to_string();
                            println!("select: Selected device '{}'", serial);
                        } else {
                            println!("select: Device not found");
                        }
                    } else {
                        println!("select: Unselected device");
                        current_sn = "".to_string();
                    }
                }

                "config" => {
                    if let Some(command) = args.next() {
                        match command {
                            "reload" => reload_config(&client, args, &current_sn),
                            "save" => save_config(&client, args, &current_sn),
                            _ => println!("config: Unknown command"),
                        }
                    } else {
                        println!("config: Unknown command");
                    }
                }

                "brightness" => {
                    if let Some(brightness) = args.next() {
                        if let Ok(brightness) = brightness.parse::<u8>() {
                            if !current_sn.is_empty() {
                                match client.set_brightness(&current_sn, brightness).expect("Failed to set brightness") {
                                    SetBrightnessResult::DeviceNotFound => println!("brightness: Device not found"),
                                    SetBrightnessResult::Set => println!("brightness: Set"),
                                }
                            } else {
                                println!("brightness: No device is selected");
                            }
                        } else {
                            println!("brightness: Input valid brightness value (0-255)");
                        }
                    } else {
                        println!("brightness: Input valid brightness value (0-255)");
                    }
                }

                _ => println!("Unknown command"),
            }
        } else {
            println!("Unknown command");
        }
    }
}

pub fn prompt_help() -> String {
    let mut help = String::new();

    for (name, help_line) in COMMANDS {
        help += &format!("{} {}\n", name.replace("-", ""), help_line.replace("[<serial>] ", ""));
    }

    help
}

// Devices
pub fn device_list(client: ClientRef) {
    let list: Vec<(bool, bool, DeviceType, String)> = client.device_list().expect("Failed to get device list")
        .into_iter()
        .map(|d| (d.online, d.managed, d.device_type, d.serial_number))
        .collect();

    // Getting table dimensions
    let mut dim: (usize, usize, usize, usize) = (6, 7, 0, 0);

    for (.., ty, serial) in &list {
        dim.2 = dim.2.max(ty.to_string().len());
        dim.3 = dim.3.max(serial.len());
    }

    // Printing
    println!("- {: <w1$} - {: <w2$} - {: <w3$} - {: <w4$} -", "Online", "Managed", "Type", "Serial", w1 = dim.0, w2 = dim.1, w3 = dim.2, w4 = dim.3);

    for (online, managed, ty, serial) in list {
        println!("| {: <w1$} | {: <w2$} | {: <w3$} | {: <w4$} |", if online {"Yes"} else {"No"}, if managed {"Yes"} else {"No"}, ty.to_string(), serial, w1 = dim.0, w2 = dim.1, w3 = dim.2, w4 = dim.3);
    }
    println!("\nTo select device for device related operations, enter 'select <serial>'");
}

pub fn add_device(client: ClientRef, mut args: Split<&str>) {
    if let Some(serial) = args.next() {
        match client.add_device(serial).expect("Failed to add device") {
            AddDeviceResult::AlreadyRegistered => println!("device add: Device is already managed"),
            AddDeviceResult::NotFound => println!("device add: Device not found"),
            AddDeviceResult::Added => println!("device add: Added to managed list"),
        }
    }
}

pub fn remove_device(client: ClientRef, mut args: Split<&str>) {
    if let Some(serial) = args.next() {
        match client.remove_device(serial).expect("Failed to add device") {
            RemoveDeviceResult::NotRegistered => println!("device remove: Device already wasn't managed"),
            RemoveDeviceResult::Removed => println!("device remove: Removed from managed list"),
        }
    }
}

pub fn reload_config(client: ClientRef, mut args: Split<&str>, current_sn: &String) {
    if let Some(arg) = args.next() {
        if arg == "all" {
            match client.reload_device_configs().expect("Failed to reload configs") {
                ReloadDeviceConfigsResult::ConfigError => println!("config reload: Error happened while reloading configs, check daemon logs"),
                ReloadDeviceConfigsResult::Reloaded => println!("config reload: Reloaded"),
            }
        } else {
            println!("config reload: Unknown operation");
        }
    } else {
        if !current_sn.is_empty() {
            match client.reload_device_config(current_sn).expect("Failed to reload config") {
                ReloadDeviceConfigResult::ConfigError => println!("config reload: Error happened while reloading config, check daemon logs"),
                ReloadDeviceConfigResult::DeviceNotFound => println!("config reload: Device not found"),
                ReloadDeviceConfigResult::Reloaded => println!("config reload: Reloaded"),
            }
        } else {
            println!("config reload: No device is selected");
        }
    }
}

pub fn save_config(client: ClientRef, mut args: Split<&str>, current_sn: &String) {
    if let Some(arg) = args.next() {
        if arg == "all" {
            match client.save_device_configs().expect("Failed to save configs") {
                SaveDeviceConfigsResult::ConfigError => println!("config save: Error happened while saving configs, check daemon logs"),
                SaveDeviceConfigsResult::Saved => println!("config save: Saved"),
            }
        } else {
            println!("config save: Unknown operation");
        }
    } else {
        if !current_sn.is_empty() {
            match client.save_device_config(current_sn).expect("Failed to save config") {
                SaveDeviceConfigResult::ConfigError => println!("config save: Error happened while saving config, check daemon logs"),
                SaveDeviceConfigResult::DeviceNotFound => println!("config save: Device not found"),
                SaveDeviceConfigResult::Saved => println!("config save: Saved"),
            }
        } else {
            println!("config save: No device is selected");
        }
    }
}