use std::cmp::min;
use crate::traits::{TWrite, TRead};

/// Dynamically sized inmemory stream.
///
/// Used as buffer with read, write and peek funcs.
pub struct Stream {
    data: Vec<u8>,
    position: usize,
}

/// Errors that can occurred if use read or peek funcs.
///
/// Inmemory stream has error EOF when try read them if it is empty.
///
/// Inmemory stream has error ZeroLength when try read them if buffer`s len is zero.
#[derive(Copy, Clone)]
#[derive(Eq, PartialEq)]
pub enum StreamError {
    EOF,
    ZeroLength,
}

impl Stream {
    /// Create new Stream with zero capacity.
    pub fn new() -> Stream {
        Stream {
            data: Vec::new(),
            position: 0,
        }
    }

    /// Create new Stream with target capacity.
    pub fn with_capacity(capacity: usize) -> Stream {
        Stream {
            data: Vec::with_capacity(capacity),
            position: 0,
        }
    }

    /// Clear bytes that have been read. Has no effect on the allocated capacity.
    pub fn truncate_readied(&mut self) {
        if self.position == self.data.len() {
            self.position = 0;
            self.data.clear();
            return;
        }
        if self.position > 0 {
            let size = self.available();
            self.data.copy_within(self.position.., 0);
            self.data.resize(size, 0);
            self.position = 0;
        }
    }

    /// Truncate the capacity of the stream as much as possible.
    pub fn truncate_capacity(&mut self) {
        self.data.shrink_to_fit();
    }
}

impl TWrite for Stream {
    /// Write one byte to stream.
    fn write_byte(&mut self, byte: u8) {
        self.data.push(byte);
    }

    /// Write bytes to stream.
    fn write(&mut self, buffer: &[u8]) -> usize {
        self.data.extend_from_slice(buffer);
        buffer.len()
    }
}

impl TRead<StreamError> for Stream {
    /// Peek one byte from stream, position don't shift.
    ///
    /// Can return EOF error.
    fn peek_byte(&self) -> Result<u8, StreamError> {
        if self.data.is_empty() {
            return Err(StreamError::EOF);
        }
        if self.position == self.data.len() {
            return Err(StreamError::EOF);
        }
        let byte = self.data[self.position];
        Ok(byte)
    }

    /// Peek bytes from stream, position don't shift.
    ///
    /// Can return EOF, ZeroLength errors.
    fn peek(&self, buffer: &mut [u8]) -> Result<usize, StreamError> {
        if self.data.is_empty() {
            return Err(StreamError::EOF);
        }
        let size = min(buffer.len(), self.available());
        if size == 0 {
            return Err(StreamError::ZeroLength);
        }
        let slice = &mut buffer[..size];
        slice.copy_from_slice(&self.data[..size]);
        Ok(size)
    }

    /// Read one byte from stream, position  shifted.
    ///
    /// Can return EOF error.
    fn read_byte(&mut self) -> Result<u8, StreamError> {
        if self.data.is_empty() {
            return Err(StreamError::EOF);
        }
        if self.position == self.data.len() {
            return Err(StreamError::EOF);
        }
        let byte = self.data[self.position];
        self.position += 1;
        Ok(byte)
    }

    /// Read bytes from stream, position shifted.
    ///
    /// Can return EOF, ZeroLength errors.
    fn read(&mut self, buffer: &mut [u8]) -> Result<usize, StreamError> {
        if self.data.is_empty() {
            return Err(StreamError::EOF);
        }
        if self.position == self.data.len() {
            return Err(StreamError::EOF);
        }
        let size = min(buffer.len(), self.available());
        if size == 0 {
            return Err(StreamError::ZeroLength);
        }
        buffer[..size].copy_from_slice(&self.data[self.position..(self.position + size)]);
        self.position += size;
        Ok(size)
    }


    /// Skip "count" bytes in stream. Return skipped bytes count.
    fn skip(&mut self, count: usize) -> usize {
        let count = min(count, self.available());
        self.position += count;
        count
    }

    /// Skip all bytes in stream. Return skipped bytes count.
    fn skip_all(&mut self) -> usize {
        let count = self.available();
        self.position += count;
        count
    }

    /// View of available bytes in stream.
    fn view(&self) -> &[u8] {
        &self.data[self.position..]
    }

    /// Return available to read bytes.
    fn available(&self) -> usize {
        self.data.len() - self.position
    }
}

