use std::cmp::min;

/// Dynamically sized inmemory stream.
///
/// Used as buffer with read, write and peek funcs.
pub struct Stream {
    data: Vec<u8>,
    position: usize,
}

/// Errors that can occurred if use read or peek funcs.
///
/// Inmemory stream has error EOF when try read them if it is empty.
///
/// Inmemory stream has error ZeroLength when try read them if buffer`s len is zero.
#[derive(Copy, Clone)]
#[derive(Eq, PartialEq)]
pub enum StreamError {
    EOF,
    ZeroLength
}

impl Stream {
    /// Create new Stream with zero capacity.
    pub fn new() -> Stream {
        Stream {
            data: Vec::new(),
            position: 0,
        }
    }

    /// Peek one byte from stream, position don't shift.
    ///
    /// Can return EOF error.
    pub fn peek_byte(&self) -> Result<u8, StreamError> {
        if self.data.is_empty() {
            return Err(StreamError::EOF);
        }
        if self.position == self.data.len() {
            return Err(StreamError::EOF);
        }
        let byte = self.data[self.position];
        Ok(byte)
    }

    /// Peek bytes from stream, position don't shift.
    ///
    /// Can return EOF, ZeroLength errors.
    pub fn peek(&self, buffer: &mut [u8]) -> Result<usize, StreamError> {
        if self.data.is_empty() {
            return Err(StreamError::EOF);
        }
        let size = min(buffer.len(), self.available());
        if size == 0 {
            return Err(StreamError::ZeroLength);
        }
        let slice = &mut buffer[..size];
        slice.copy_from_slice(&self.data[..size]);
        Ok(size)
    }

    /// Read one byte from stream, position  shifted.
    ///
    /// Can return EOF error.
    pub fn read_byte(&mut self) -> Result<u8, StreamError> {
        if self.data.is_empty() {
            return Err(StreamError::EOF);
        }
        if self.position == self.data.len() {
            return Err(StreamError::EOF);
        }
        let byte = self.data[self.position];
        self.position += 1;
        Ok(byte)
    }

    /// Read bytes from stream, position shifted.
    ///
    /// Can return EOF, ZeroLength errors.
    pub fn read(&mut self, buffer: &mut [u8]) -> Result<usize, StreamError> {
        if self.data.is_empty() {
            return Err(StreamError::EOF);
        }
        if self.position == self.data.len() {
            return Err(StreamError::EOF);
        }
        let size = min(buffer.len(), self.available());
        if size == 0 {
            return Err(StreamError::ZeroLength);
        }
        buffer[..size].copy_from_slice(&self.data[self.position..(self.position + size)]);
        self.position += size;
        Ok(size)
    }

    /// Write one byte to stream.
    pub fn write_byte(&mut self, byte: u8) {
        self.data.push(byte);
    }

    /// Write bytes to stream.
    pub fn write(&mut self, buffer: &[u8]) -> usize {
        self.data.extend_from_slice(buffer);
        buffer.len()
    }

    /// Clear bytes that have been read. Has no effect on the allocated capacity.
    pub fn truncate_readied(&mut self) {
        if self.position > 0 {
            self.data.copy_within(self.position..,0);
            self.position = 0;
        }
    }

    /// Truncate the capacity of the stream as much as possible.
    pub fn truncate_capacity(&mut self) {
        self.data.shrink_to_fit();
    }

    /// Return available to read bytes.
    pub fn available(&self) -> usize {
        self.data.len() - self.position
    }
}

