use crate::stream::{Stream, StreamError};

///Reader for inmemory stream
#[derive(Debug)]
pub struct Reader<'a> {
    stream: &'a mut Stream,
}

///Writer for inmemory stream
#[derive(Debug)]
pub struct Writer<'a> {
    stream: &'a mut Stream,
}

impl<'a> Reader<'a> {
    /// Create new Reader based on stream.
    pub fn new(stream: &mut Stream) -> Reader {
        Reader {
            stream
        }
    }

    /// Peek one byte from stream, position don't shift.
    ///
    /// Can return EOF error.
    pub fn peek_byte(&self) -> Result<u8, StreamError> {
        self.stream.peek_byte()
    }

    /// Peek bytes from stream, position don't shift.
    ///
    /// Can return EOF, ZeroLength errors.
    pub fn peek(&self, buffer: &mut [u8]) -> Result<usize, StreamError> {
        self.stream.peek(buffer)
    }

    /// Read one byte from stream, position  shifted.
    ///
    /// Can return EOF error.
    pub fn read_byte(&mut self) -> Result<u8, StreamError> {
        self.stream.read_byte()
    }

    /// Read bytes from stream, position shifted.
    ///
    /// Can return EOF, ZeroLength errors.
    pub fn read(&mut self, buffer: &mut [u8]) -> Result<usize, StreamError> {
        self.stream.read(buffer)
    }

    /// Return available to read bytes.
    pub fn available(&self) -> usize {
        self.stream.available()
    }

    /// Clear bytes that have been read. Has no effect on the allocated capacity.
    pub fn truncate_readied(&mut self) {
        self.stream.truncate_readied();
    }

    /// Truncate the capacity of the stream as much as possible.
    pub fn truncate_capacity(&mut self) {
        self.stream.truncate_capacity();
    }
}

impl<'a> Writer<'a> {
    /// Create new Writer based on stream.
    pub fn new(stream: &mut Stream) -> Writer {
        Writer {
            stream
        }
    }

    /// Write one byte to stream.
    pub fn write_byte(&mut self, byte: u8) {
        self.stream.write_byte(byte);
    }

    /// Write bytes to stream.
    pub fn write(&mut self, buffer: &[u8]) -> usize {
        self.stream.write(buffer)
    }
}