use crate::{Error, Result};
use async_std::sync::{Arc, Mutex};
use extended_primitives::Buffer;
use serde::{Deserialize, Serialize};
use std::fmt;

#[derive(Clone, Default)]
pub struct ReadyIndicator(Arc<Mutex<bool>>);

impl ReadyIndicator {
    pub fn new(ready: bool) -> Self {
        ReadyIndicator(Arc::new(Mutex::new(ready)))
    }

    pub async fn ready(&self) {
        *self.0.lock().await = true;
    }

    pub async fn not_ready(&self) {
        *self.0.lock().await = false;
    }

    pub async fn inner(&self) -> bool {
        *self.0.lock().await
    }
}

#[derive(Serialize, Deserialize, Clone, Debug)]
#[serde(untagged)]
pub enum ID {
    Num(u64),
    Str(String),
    Null(serde_json::Value),
}

impl std::fmt::Display for ID {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        match self {
            ID::Num(ref e) => write!(f, "{}", e),
            ID::Str(ref e) => write!(f, "{}", e),
            ID::Null(ref _e) => write!(f, "null"),
        }
    }
}

pub enum MessageValue {
    StratumV1(serde_json::map::Map<String, serde_json::Value>),
    ExMessage(ExMessageGeneric),
}

pub enum MessageTypes {
    RegisterWorker,
    SubmitShare,
    SubmitShareWithTime,
    SubmitShareWithVersion,
    SubmitShareWithTimeAndVersion,
    UnregisterWorker,
    MiningSetDiff,

    Unknown(u8),
}

impl MessageTypes {
    pub fn from_u8(cmd: u8) -> Self {
        match cmd {
            0x01 => MessageTypes::RegisterWorker,
            0x02 => MessageTypes::SubmitShare,
            0x03 => MessageTypes::SubmitShareWithTime,
            0x04 => MessageTypes::UnregisterWorker,
            0x05 => MessageTypes::MiningSetDiff,
            //@note not sure why these are so far after the originals. Makes me think there are
            //other messages we are missing here, but can figure that out later.
            0x12 => MessageTypes::SubmitShareWithVersion,
            0x13 => MessageTypes::SubmitShareWithTimeAndVersion,
            _ => MessageTypes::Unknown(cmd),
        }
    }

    pub fn to_string(&self) -> String {
        match *self {
            MessageTypes::RegisterWorker => String::from("exMessageRegisterWorker"),
            MessageTypes::SubmitShare => String::from("exMessageSubmitShare"),
            MessageTypes::SubmitShareWithTime => String::from("exMessageSubmitShare"),
            MessageTypes::UnregisterWorker => String::from("exMessageUnregisterWorker"),
            MessageTypes::MiningSetDiff => String::from("exMessageMiningSetDiff"),
            MessageTypes::SubmitShareWithVersion => String::from("exMessageSubmitShare"),
            MessageTypes::SubmitShareWithTimeAndVersion => String::from("exMessageSubmitShare"),
            //@todo I'd actually prefer we throw an error here but we can come back to that later
            //as well.
            //But seems important for security otherwise someone could just hit us with fake Ex
            //messages all day.
            _ => String::from(""),
        }
    }
}

pub struct ExMessageGeneric {
    pub magic_number: u8,
    //@todo turn this into an Enum?
    pub cmd: MessageTypes,
    pub length: u16,
    pub body: Buffer,
}

//@todo let's make sure we get some good errors in here for if this shit blows up.
impl ExMessageGeneric {
    //@todo actually just call this inside of that function.
    pub fn from_buffer(buffer: &mut Buffer) -> Result<Self> {
        let magic_number = buffer.read_u8().map_err(|_| Error::BrokenExHeader)?;
        let cmd = buffer.read_u8().map_err(|_| Error::BrokenExHeader)?;
        let length = buffer.read_u16().map_err(|_| Error::BrokenExHeader)?;

        let body = buffer.clone();

        let cmd = MessageTypes::from_u8(cmd);

        //@todo I think we need some kind of sanity checks with length and the body here, but I'd
        //rather let a few bad messages slip than weed out good ones rn.

        // if cmd == MessageTypes::Unknown(_) {
        //@todo custom error here pls.
        // return Error::
        // }

        Ok(ExMessageGeneric {
            magic_number,
            cmd,
            length,
            body,
        })
    }
}
