pub use crate::connection::Connection;
use crate::Result;
use async_std::sync::{Arc, RwLock};
use std::collections::HashMap;
use std::net::SocketAddr;

#[derive(Default)]
pub struct MinerList<CState: Clone + Sync + Send + 'static> {
    //@todo there are faster data structures than hashmap. Investigate using some of those.
    pub miners: RwLock<HashMap<SocketAddr, Arc<Connection<CState>>>>,
    pub max_connections: Option<usize>,
}

impl<CState: Clone + Sync + Send + 'static> MinerList<CState> {
    pub fn new(max_connections: Option<usize>) -> Self {
        MinerList {
            miners: RwLock::new(HashMap::new()),
            max_connections,
        }
    }

    pub async fn add_miner(&self, addr: SocketAddr, miner: Arc<Connection<CState>>) -> Result<()> {
        self.miners.write().await.insert(addr, miner);
        // gauge!(
        //     "stratum.num_connections",
        //     self.miners.read().await.len() as f64
        // );
        Ok(())
    }

    pub async fn remove_miner(&self, addr: SocketAddr) -> Result<()> {
        self.miners.write().await.remove(&addr);
        // gauge!(
        //     "stratum.num_connections",
        //     self.miners.read().await.len() as f64
        // );
        Ok(())
    }

    pub async fn get_all_miners(&self) -> Vec<Arc<Connection<CState>>> {
        self.miners.read().await.values().cloned().collect()
    }

    pub async fn len(&self) -> usize {
        self.miners.read().await.len()
    }

    pub async fn is_empty(&self) -> bool {
        self.miners.read().await.is_empty()
    }

    pub async fn is_full(&self) -> bool {
        if let Some(max) = self.max_connections {
            if self.len().await < max {
                false
            } else {
                true
            }
        } else {
            false
        }
    }

    pub async fn shutdown(&self) -> Result<()> {
        //@todo we need to parallize this async.
        for miner in self.miners.read().await.values() {
            miner.shutdown().await?;
        }

        Ok(())
    }
}
